package com.atlassian.crowd.directory.ldap.credential;

import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.InvalidCredentialException;

/**
 * Encodes and/or encrypts a given {@link PasswordCredential} to the value expected by a particular directory connector.
 *
 * This interface should not be implemented directly; implement {@link LDAPCredentialToStringEncoder} or
 * {@link LDAPCredentialToByteArrayEncoder} instead.
 */
public interface LDAPCredentialEncoder {
    /**
     * Given an plain-text password, encodes/encrypts it according to the settings required by the particular directory
     * connector which this encoder is intended to be used with. Return type MUST be either String or byte[].
     *
     * @param passwordCredential The password credential to be encoded, guanteed to be non-null. If the subclass
     *                           supports encrypting the given passwordCredential, it should NOT do so if
     *                           {@link com.atlassian.crowd.embedded.api.PasswordCredential#isEncryptedCredential()}
     *                           returns {@code true} (although it may still <em>encode</em> the credential to the format
     *                           required by the directory). The value of
     *                           {@link com.atlassian.crowd.embedded.api.PasswordCredential#getCredential()} is also
     *                           guaranteed to be non-null.
     * @return An encoded password, suitable for passing to the directory.
     * @throws InvalidCredentialException If the password could not be encoded.
     */
    Object encodeCredential(PasswordCredential passwordCredential) throws InvalidCredentialException;

    /**
     * @return {@code true} if the LDAP directory supports client-side encryption of passwords.
     */
    boolean supportsSettingEncryptedPasswords();

    public interface LDAPCredentialToStringEncoder extends LDAPCredentialEncoder {
        /**
         * @return String instead of Object
         * @see LDAPCredentialEncoder#encodeCredential(com.atlassian.crowd.embedded.api.PasswordCredential)
         */
        @Override
        String encodeCredential(PasswordCredential passwordCredential) throws InvalidCredentialException;
    }

    public interface LDAPCredentialToByteArrayEncoder extends LDAPCredentialEncoder {
        /**
         * @return byte[] instead of Object
         * @see LDAPCredentialEncoder#encodeCredential(com.atlassian.crowd.embedded.api.PasswordCredential)
         */
        @Override
        byte[] encodeCredential(PasswordCredential passwordCredential) throws InvalidCredentialException;
    }
}
