package com.atlassian.crowd.directory.ldap.credential;

import com.atlassian.crowd.directory.ldap.credential.LDAPCredentialEncoder.LDAPCredentialToStringEncoder;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.util.PasswordHelper;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * An encoder which passes a non-encrypted {@link PasswordCredential}'s credential through untouched, but throws an
 * exception for every already-encrypted  (i.e. when {@link PasswordCredential#isEncryptedCredential()} returns {@code true}).
 *
 * The exception is {@link PasswordCredential#NONE}, in which case it generates a random password and returns that instead.
 */
public class EnforceUnencryptedCredentialEncoder implements LDAPCredentialToStringEncoder {
    private final PasswordHelper passwordHelper;

    public EnforceUnencryptedCredentialEncoder(PasswordHelper passwordHelper) {
        this.passwordHelper = checkNotNull(passwordHelper);
    }

    /**
     * @throws InvalidCredentialException if {@link com.atlassian.crowd.embedded.api.PasswordCredential#isEncryptedCredential()}
     *                                    returns true AND the given passwordCredential is not equal to {@link PasswordCredential#NONE}
     */
    @Override
    public String encodeCredential(PasswordCredential passwordCredential) throws InvalidCredentialException {
        if (PasswordCredential.NONE.equals(passwordCredential)) {
            // since we don't support setting passwords by hash, the best we can do is generate a random password instead
            return passwordHelper.generateRandomPassword();
        } else if (passwordCredential.isEncryptedCredential()) {
            throw new InvalidCredentialException("Setting already encrypted passwords is not supported in this directory");
        } else {
            return passwordCredential.getCredential();
        }
    }

    @Override
    public boolean supportsSettingEncryptedPasswords() {
        return false;
    }

    public PasswordHelper getPasswordHelper() {
        return passwordHelper;
    }
}
