package com.atlassian.crowd.directory.ldap.cache;

import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.synchronisation.CacheSynchronisationResult;
import com.atlassian.crowd.directory.synchronisation.PartialSynchronisationResult;
import com.atlassian.crowd.directory.synchronisation.cache.AbstractCacheRefresher;
import com.atlassian.crowd.directory.synchronisation.cache.CacheRefresher;
import com.atlassian.crowd.directory.synchronisation.cache.DirectoryCache;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.DirectoryEntities;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplateWithAttributes;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * A simple implementation of CacheRefresher that will only do "Full Refresh".
 * This is used for all LDAP servers other than AD.
 *
 * @since v2.1
 */
public class RemoteDirectoryCacheRefresher extends AbstractCacheRefresher implements CacheRefresher {
    private static final Logger log = LoggerFactory.getLogger(RemoteDirectoryCacheRefresher.class);

    public RemoteDirectoryCacheRefresher(final RemoteDirectory remoteDirectory) {
        super(remoteDirectory);
    }

    public CacheSynchronisationResult synchroniseChanges(final DirectoryCache directoryCache, String syncToken) throws OperationFailedException {
        // We can never do a delta sync
        return CacheSynchronisationResult.FAILURE;
    }

    protected List<UserWithAttributes> findAllRemoteUsers(final boolean withAttributes) throws OperationFailedException {
        final long start = System.currentTimeMillis();
        log.debug("loading remote users");
        final List<UserWithAttributes> users;
        if (withAttributes) {
            users = remoteDirectory.searchUsers(getUserQuery(UserWithAttributes.class));
        } else {
            users = remoteDirectory.searchUsers(getUserQuery(User.class)).stream()
                    .map(UserTemplateWithAttributes::toUserWithNoAttributes)
                    .collect(Collectors.toList());
        }
        log.info("found [ {} ] remote users in [ {} ms ]", users.size(), (System.currentTimeMillis() - start));
        return users;
    }

    private <T extends User> EntityQuery<T> getUserQuery(final Class<T> clazz) {
        return QueryBuilder.queryFor(clazz, EntityDescriptor.user())
                .returningAtMost(EntityQuery.ALL_RESULTS);
    }

    private <T extends Group> EntityQuery<T> getGroupQuery(final Class<T> clazz) {
        return QueryBuilder.queryFor(clazz, EntityDescriptor.group(GroupType.GROUP)).returningAtMost(EntityQuery.ALL_RESULTS);
    }

    protected List<GroupWithAttributes> findAllRemoteGroups(boolean withAttributes) throws OperationFailedException {
        final long start = System.currentTimeMillis();
        log.debug("loading remote groups");
        final List<GroupWithAttributes> groups;
        if (withAttributes) {
            groups = remoteDirectory.searchGroups(getGroupQuery(GroupWithAttributes.class));
        } else {
            groups = remoteDirectory.searchGroups(getGroupQuery(Group.class)).stream()
                    .map(GroupTemplateWithAttributes::ofGroupWithNoAttributes)
                    .collect(Collectors.toList());
        }
        log.info("found [ {} ] remote groups in [ {} ms ]", groups.size(), (System.currentTimeMillis() - start));
        return groups;
    }

    @Override
    protected PartialSynchronisationResult<? extends UserWithAttributes> synchroniseAllUsers(DirectoryCache directoryCache) throws OperationFailedException {
        final Date syncStartDate = new Date();

        final List<? extends UserWithAttributes> ldapUsers = findAllRemoteUsers(isUserAttributeSynchronisationEnabled());

        // By removing missing users before adding/updating, users that have been renamed can be tracked correctly.
        // See JDEV-22181 for a more in-depth discussion.
        directoryCache.deleteCachedUsersNotIn(ldapUsers, syncStartDate);
        directoryCache.addOrUpdateCachedUsers(ldapUsers, syncStartDate);

        return new PartialSynchronisationResult<>(ldapUsers);
    }

    @Override
    protected PartialSynchronisationResult<? extends GroupWithAttributes> synchroniseAllGroups(final DirectoryCache directoryCache) throws OperationFailedException {
        final Date syncStartDate = new Date();

        //CWD-2504: We filter out the duplicate groups. This basically means that these groups don't exist to crowd.
        // Crowd can't currently deal with groups with the same name so rather than throwing a runtime exception
        // which kills the synchronization, we just ignore them and log a message and continue to work.
        // The admins of the LDAP directory will have to do some admin to get the groups to work correctly.
        final List<? extends GroupWithAttributes> groups = DirectoryEntities.filterOutDuplicates(findAllRemoteGroups(isGroupAttributeSynchronisationEnabled()));

        // By removing missing groups before adding/updating, groups that have been renamed in a way that looks the same
        // to the database will be updated in one sync instead of needing two.
        // See CWD-4290 for the motivating example.
        directoryCache.deleteCachedGroupsNotIn(GroupType.GROUP, groups, syncStartDate);
        directoryCache.addOrUpdateCachedGroups(groups, syncStartDate);

        return new PartialSynchronisationResult<>(groups);
    }
}
