package com.atlassian.crowd.directory.ldap.cache;

import javax.annotation.Nullable;

import com.atlassian.crowd.directory.RemoteCrowdDirectory;
import com.atlassian.crowd.directory.synchronisation.CacheSynchronisationResult;
import com.atlassian.crowd.directory.synchronisation.PartialSynchronisationResult;
import com.atlassian.crowd.directory.synchronisation.cache.AbstractCacheRefresher;
import com.atlassian.crowd.directory.synchronisation.cache.CacheRefresher;
import com.atlassian.crowd.directory.synchronisation.cache.DirectoryCache;
import com.atlassian.crowd.event.EventTokenExpiredException;
import com.atlassian.crowd.event.Events;
import com.atlassian.crowd.event.IncrementalSynchronisationNotAvailableException;
import com.atlassian.crowd.exception.CrowdException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UnsupportedCrowdApiException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.event.GroupEvent;
import com.atlassian.crowd.model.event.GroupMembershipEvent;
import com.atlassian.crowd.model.event.Operation;
import com.atlassian.crowd.model.event.OperationEvent;
import com.atlassian.crowd.model.event.UserEvent;
import com.atlassian.crowd.model.event.UserMembershipEvent;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;

import com.google.common.base.Strings;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EventTokenChangedCacheRefresher extends AbstractCacheRefresher {
    private static final Logger log = LoggerFactory.getLogger(EventTokenChangedCacheRefresher.class);

    private final CacheRefresher fullSyncCacheRefresher;

    private final RemoteCrowdDirectory crowdDirectory;

    public EventTokenChangedCacheRefresher(final RemoteCrowdDirectory crowdDirectory, CacheRefresher fullSyncCacheRefresher) {
        super(crowdDirectory);
        this.crowdDirectory = crowdDirectory;
        this.fullSyncCacheRefresher = fullSyncCacheRefresher;
    }

    @Override
    public CacheSynchronisationResult synchroniseAll(DirectoryCache directoryCache) throws OperationFailedException {
        String initialEventToken = null;

        if (isIncrementalSyncEnabled()) {
            try {
                initialEventToken = crowdDirectory.getCurrentEventToken();
            } catch (UnsupportedCrowdApiException e) {
                log.debug("Remote server does not support event based sync.");
            } catch (OperationFailedException e) {
                log.warn("Could not update event token.", e);
            } catch (IncrementalSynchronisationNotAvailableException e) {
                log.warn("Incremental synchronisation is not available. Falling back to full synchronisation", e);
            }
        }

        final CacheSynchronisationResult result = fullSyncCacheRefresher.synchroniseAll(directoryCache);
        String postEventToken = null;
        try {
            postEventToken = crowdDirectory.getCurrentEventToken();
            if (initialEventToken != null) {
                if (!initialEventToken.equals(postEventToken)) {
                    log.warn("Possible events during full synchronisation");
                }
            }
        } catch (CrowdException e) {
            log.debug("Failed to retrieve event token after full synchronisation", e);
        }

        if (result.isSuccess()) {
            return new CacheSynchronisationResult(true, postEventToken);
        }
        return CacheSynchronisationResult.FAILURE;
    }

    @Override
    public CacheSynchronisationResult synchroniseChanges(DirectoryCache directoryCache, @Nullable String eventToken) throws OperationFailedException {
        if (!isIncrementalSyncEnabled()) {
            log.debug("Incremental synchronisation is not enabled");
            return CacheSynchronisationResult.FAILURE;
        }


        if (Strings.emptyToNull(eventToken) == null) {
            log.debug("A full synchronisation is needed to obtain the current token event");
            return CacheSynchronisationResult.FAILURE;
        }

        final Events events;
        try {
            events = crowdDirectory.getNewEvents(eventToken);
        } catch (EventTokenExpiredException e) {
            if (e.getMessage() != null) {
                log.error("Incremental synchronisation failed: {}", e.getMessage());
            }
            return CacheSynchronisationResult.FAILURE;
        }

        for (OperationEvent event : events.getEvents()) {
            if (event instanceof UserEvent) {
                final UserEvent userEvent = (UserEvent) event;
                if (event.getOperation() == Operation.CREATED || event.getOperation() == Operation.UPDATED) {
                    directoryCache.addOrUpdateCachedUser(userEvent.getUser());
                    if (isUserAttributeSynchronisationEnabled()) {
                        try {
                            directoryCache.applySyncingUserAttributes(userEvent.getUser().getName(),
                                    userEvent.getDeletedAttributes(),
                                    userEvent.getStoredAttributes());
                        } catch (UserNotFoundException e) {
                            throw new OperationFailedException("Failed to synchronize directory user attributes for missing user: " + userEvent.getUser().getName());
                        }
                    }
                } else if (event.getOperation() == Operation.DELETED) {
                    directoryCache.deleteCachedUser(userEvent.getUser().getName());
                }
            } else if (event instanceof GroupEvent) {
                final GroupEvent groupEvent = (GroupEvent) event;
                if (event.getOperation() == Operation.CREATED || event.getOperation() == Operation.UPDATED) {
                    directoryCache.addOrUpdateCachedGroup(groupEvent.getGroup());
                    try {
                        directoryCache.applySyncingGroupAttributes(groupEvent.getGroup().getName(),
                                groupEvent.getDeletedAttributes(),
                                groupEvent.getStoredAttributes());
                    } catch (GroupNotFoundException e) {
                        throw new OperationFailedException("Failed to synchronize directory group attributes for missing group: " + groupEvent.getGroup().getName());
                    }
                } else if (event.getOperation() == Operation.DELETED) {
                    directoryCache.deleteCachedGroup(groupEvent.getGroup().getName());
                }
            } else if (event instanceof UserMembershipEvent) {
                UserMembershipEvent membershipEvent = (UserMembershipEvent) event;
                if (event.getOperation() == Operation.CREATED) {
                    for (String parentGroupName : membershipEvent.getParentGroupNames()) {
                        directoryCache.addUserToGroup(membershipEvent.getChildUsername(), parentGroupName);
                    }
                } else if (event.getOperation() == Operation.DELETED) {
                    for (String parentGroupName : membershipEvent.getParentGroupNames()) {
                        directoryCache.removeUserFromGroup(membershipEvent.getChildUsername(), parentGroupName);
                    }
                } else if (event.getOperation() == Operation.UPDATED) {
                    directoryCache.syncGroupMembershipsForUser(membershipEvent.getChildUsername(), membershipEvent.getParentGroupNames());
                }
            } else if (event instanceof GroupMembershipEvent) {
                GroupMembershipEvent membershipEvent = (GroupMembershipEvent) event;
                if (event.getOperation() == Operation.CREATED) {
                    for (String parentGroupName : membershipEvent.getParentGroupNames()) {
                        directoryCache.addGroupToGroup(membershipEvent.getGroupName(), parentGroupName);
                    }
                } else if (event.getOperation() == Operation.DELETED) {
                    for (String parentGroupName : membershipEvent.getParentGroupNames()) {
                        directoryCache.removeGroupFromGroup(membershipEvent.getGroupName(), parentGroupName);
                    }
                } else if (event.getOperation() == Operation.UPDATED) {
                    directoryCache.syncGroupMembershipsAndMembersForGroup(membershipEvent.getGroupName(), membershipEvent.getParentGroupNames(), membershipEvent.getChildGroupNames());
                }
            } else {
                throw new RuntimeException("Unsupported event " + event);
            }
        }

        // only if the incremental sync has completed, we may use the event token for the next incremental sync
        return new CacheSynchronisationResult(true, events.getNewEventToken());
    }

    @Override
    protected PartialSynchronisationResult<? extends UserWithAttributes> synchroniseAllUsers(DirectoryCache directoryCache) throws OperationFailedException {
        throw new UnsupportedOperationException();
    }

    @Override
    protected PartialSynchronisationResult<? extends GroupWithAttributes> synchroniseAllGroups(DirectoryCache directoryCache) throws OperationFailedException {
        throw new UnsupportedOperationException();
    }
}
