package com.atlassian.crowd.directory.ldap;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;

import javax.xml.bind.annotation.adapters.XmlAdapter;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;
import java.net.URI;
import java.net.URISyntaxException;

/**
 * Enum class for LDAP security modes.
 */
@XmlJavaTypeAdapter(LdapSecureMode.Adapter.class)
public enum LdapSecureMode {
    NONE("false", "ldap", 389), LDAPS("true", "ldaps", 636), START_TLS("starttls", "ldap", 389);

    private static final Splitter SPACE_SPLITTER = Splitter.on(" ").trimResults().omitEmptyStrings();

    private String name;
    private String protocol;
    private int defaultPort;

    LdapSecureMode(String name, String protocol, int defaultPort) {
        this.name = name;
        this.protocol = protocol;
        this.defaultPort = defaultPort;
    }

    @Override
    public String toString() {
        return name;
    }

    public String getName() {
        return name;
    }

    public String getProtocol() {
        return protocol;
    }

    public int getDefaultPort() {
        return defaultPort;
    }

    public static LdapSecureMode fromString(String name) {
        // Backward compatible support for booleans.
        for (LdapSecureMode mode : values()) {
            if (mode.getName().equalsIgnoreCase(name)) {
                return mode;
            }
        }
        return NONE;
    }

    public static LdapSecureMode fromUrl(String url) {
        Iterable<String> urls = SPACE_SPLITTER.split(Strings.nullToEmpty(url));
        if (urls.iterator().hasNext()) {
            try {
                URI uri = new URI(urls.iterator().next());
                if (LDAPS.getProtocol().equalsIgnoreCase(uri.getScheme())) {
                    return LDAPS;
                }
            } catch (URISyntaxException e) {
                throw new RuntimeException(e);
            }
        }
        return LdapSecureMode.NONE;
    }

    public static class Adapter extends XmlAdapter<String, LdapSecureMode> {

        public String marshal(LdapSecureMode qualifier) {
            return qualifier.getName();
        }

        public LdapSecureMode unmarshal(String val) {
            return LdapSecureMode.fromString(val);
        }
    }
}
