package com.atlassian.crowd.directory.hybrid;

import com.atlassian.crowd.directory.InternalRemoteDirectory;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.InternalDirectoryGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Manages local group creation and mutation.
 * <p>
 * A local group is an accessible group that does NOT exist
 * in LDAP.  If local groups is enabled, then all mutation
 * operations execute on local groups.
 * <p>
 * Any group in the internal directory section of a
 * {@link com.atlassian.crowd.directory.DbCachingRemoteDirectory}
 * with the shadow attribute set to "false" is an local group.
 */
public class LocalGroupHandler extends InternalGroupHandler {
    private static final Logger log = LoggerFactory.getLogger(LocalGroupHandler.class);

    public LocalGroupHandler(InternalRemoteDirectory internalDirectory) {
        super(internalDirectory);
    }

    /**
     * Finds a local group.
     *
     * @param groupName name of group.
     * @return local group.
     * @throws GroupNotFoundException local group with supplied name does not exist.
     * @throws ReadOnlyGroupException when the group exists in cache but is not local.
     */
    public Group findLocalGroup(String groupName) throws GroupNotFoundException, ReadOnlyGroupException {
        InternalDirectoryGroup group = getInternalDirectory().findGroupByName(groupName);

        if (group.isLocal()) {
            return group;
        } else {
            throw new ReadOnlyGroupException(groupName);
        }
    }

    /**
     * Creates a local group with the supplied template.
     * <p>
     * NOTE: if a local group with the same name of groupTemplate already exists,
     * then the underlying directory may throw an Exception
     * (in accordance with the InternalDirectory.addGroup implementations).
     *
     * @param groupTemplate group to add.
     * @return added group.
     * @throws com.atlassian.crowd.exception.InvalidGroupException    group already exists, either as a local group or as a shadow, with the same name.
     * @throws com.atlassian.crowd.exception.OperationFailedException underlying directory implementation failed to execute the operation.
     */
    public Group createLocalGroup(GroupTemplate groupTemplate)
            throws InvalidGroupException, OperationFailedException, DirectoryNotFoundException {
        return getInternalDirectory().addLocalGroup(groupTemplate);
    }

    /**
     * Updates a local group.
     * <p>
     * If the group found is shadow group or does not exist, a GroupNotFoundException will be thrown.
     *
     * @param groupTemplate group to update.
     * @return updated group.
     * @throws com.atlassian.crowd.exception.OperationFailedException underlying directory implementation failed to execute the operation.
     */
    public Group updateLocalGroup(GroupTemplate groupTemplate)
            throws OperationFailedException, GroupNotFoundException, ReadOnlyGroupException, InvalidGroupException {
        findLocalGroup(groupTemplate.getName()); // throws gnfe if no local group with supplied name found

        return getInternalDirectory().updateGroup(groupTemplate);
    }

    public void addUserToLocalGroup(String username, String groupName)
            throws OperationFailedException, GroupNotFoundException, ReadOnlyGroupException, UserNotFoundException,
            MembershipAlreadyExistsException {
        findLocalGroup(groupName);
        getInternalDirectory().addUserToGroup(username, groupName);
    }

    public void removeUserFromLocalGroup(String username, String groupName)
            throws OperationFailedException, GroupNotFoundException, MembershipNotFoundException, ReadOnlyGroupException, UserNotFoundException {
        findLocalGroup(groupName); // throws gnfe

        getInternalDirectory().removeUserFromGroup(username, groupName); // throws unfe, mnfe
    }

    public void addGroupToGroup(String parentGroup, String childGroup) throws GroupNotFoundException, OperationFailedException, ReadOnlyGroupException, MembershipAlreadyExistsException, InvalidMembershipException {
        findLocalGroup(parentGroup);
        try {
            getInternalDirectory().addGroupToGroup(childGroup, parentGroup);
        } catch (MembershipAlreadyExistsException e) {
            // swallow exception as group is already in group
            log.debug("Group ({}) is already a member of group ({}).", childGroup, parentGroup);
        }
    }

    public void removeGroupFromGroup(String childGroup, String parentGroup) throws GroupNotFoundException, OperationFailedException, ReadOnlyGroupException, MembershipNotFoundException, InvalidMembershipException {
        findLocalGroup(parentGroup);
        getInternalDirectory().removeGroupFromGroup(childGroup, parentGroup);
    }
}
