package com.atlassian.crowd.directory;

import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.ldap.core.ContextSource;
import org.springframework.ldap.pool2.factory.PooledContextSource;

import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;

public class SpringLdapPooledContextSourceProvider implements DisposableBean, SpringLdapPoolStatisticsProvider {
    private static final Logger logger = LoggerFactory.getLogger(SpringLdapPooledContextSourceProvider.class);

    private final LdapContextSourceFactory ldapContextSourceFactory;
    private final ConcurrentMap<Long, ContextSourceEntry> contextSources;

    public SpringLdapPooledContextSourceProvider(final LdapContextSourceFactory ldapContextSourceFactory) {
        this.ldapContextSourceFactory = ldapContextSourceFactory;
        contextSources = Maps.newConcurrentMap();
    }

    @VisibleForTesting
    SpringLdapPooledContextSourceProvider(final LdapContextSourceFactory ldapContextSourceFactory, final ConcurrentMap<Long, ContextSourceEntry> contextSources) {
        this.ldapContextSourceFactory = ldapContextSourceFactory;
        this.contextSources = contextSources;
    }

    /**
     * Returns pooled Ldap ContextSource for specified directoryId. If current pool's parameters are different than passed, then pool is reinitialized.
     */
    ContextSource getContextSource(final long directoryId, final LDAPPropertiesMapper ldapPropertiesMapper, final Map<String, Object> envProperties) {
        ContextSourceEntry contestSourceEntry = contextSources.get(directoryId);

        if (contestSourceEntry != null && hasSameAttributesAndProperties(contestSourceEntry, ldapPropertiesMapper, envProperties)) {
            return contestSourceEntry.contextSource;
        }

        return contextSources.compute(directoryId, (id, existingEntry) -> mergeEntries(directoryId, existingEntry, ldapPropertiesMapper, envProperties)).contextSource;
    }

    private ContextSourceEntry mergeEntries(long directoryId, final ContextSourceEntry existingEntry, final LDAPPropertiesMapper ldapPropertiesMapper, final Map<String, Object> envProperties) {
        if (existingEntry != null) {
           if (hasSameAttributesAndProperties(existingEntry, ldapPropertiesMapper, envProperties)) {
               return existingEntry;
           } else {
               try {
                   existingEntry.contextSource.destroy();
               } catch (Exception e) {
                   logger.error("Failed to destroy contextSource", e);
               }
           }
        }

        logger.debug("Allocating new ldap pool for directory {}", directoryId);
        final PooledContextSource contextSource = ldapContextSourceFactory.createPooledContextSource(directoryId, ldapPropertiesMapper, envProperties);

        return new ContextSourceEntry(contextSource, ldapPropertiesMapper, envProperties);
    }

    @Override
    public void destroy() throws Exception {
        Exception lastException = null;

        for (ContextSourceEntry value : contextSources.values()) {
            try {
                value.contextSource.destroy();
            } catch (Exception e) {
                lastException = e;
            }
        }

        if (lastException != null) {
            throw lastException;
        }
    }

    @Override
    public Map<Long, SpringLdapPoolStatistics> getPoolStatistics() {
        return contextSources.entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey,
                        entry -> SpringLdapPoolStatistics.fromPool(entry.getValue().contextSource)));
    }

    private boolean hasSameAttributesAndProperties(ContextSourceEntry contextSourceEntry, LDAPPropertiesMapper attributes, Map<String, Object> envProperties) {
        return ldapContextSourceFactory.areConnectionPropertiesSame(attributes, contextSourceEntry.attributes)
                && contextSourceEntry.envProperties.equals(envProperties);
    }

    @VisibleForTesting
    static class ContextSourceEntry {
        private final PooledContextSource contextSource;
        private final LDAPPropertiesMapper attributes;
        private final Map<String, Object> envProperties;

        ContextSourceEntry(PooledContextSource contextSource, LDAPPropertiesMapper attributes, Map<String, Object> envProperties) {
            this.contextSource = contextSource;
            this.attributes = attributes;
            this.envProperties = envProperties;
        }
    }

}
