package com.atlassian.crowd.console.value.directory;

import com.atlassian.crowd.directory.SynchronisableDirectoryProperties;
import com.atlassian.crowd.directory.monitor.poller.PollerConfig;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.model.directory.DirectoryImpl;

import java.util.Map;
import java.util.TimeZone;

/**
 * Model for common settings for connector and crowd directories that can do synchronisation.
 */
public abstract class SynchronisableDirectoryConnection {
    private final String serverTimeZone = TimeZone.getDefault().getID();
    private SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth syncGroupMembershipsAfterAuth = SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.DEFAULT;
    private boolean incrementalSyncEnabled = false;
    private PollerConfig pollerConfig = new PollerConfig();

    public void loadFromDirectory(Directory directory) {
        incrementalSyncEnabled = Boolean.parseBoolean(directory.getValue(SynchronisableDirectoryProperties.INCREMENTAL_SYNC_ENABLED));
        pollerConfig.copyFrom(directory.getAttributes());
        syncGroupMembershipsAfterAuth = SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.forDirectory(directory);
    }

    public void updateDirectory(DirectoryImpl directory) {
        directory.setAttribute(SynchronisableDirectoryProperties.INCREMENTAL_SYNC_ENABLED, Boolean.toString(incrementalSyncEnabled));

        directory.setAttribute(SynchronisableDirectoryProperties.SYNC_GROUP_MEMBERSHIP_AFTER_SUCCESSFUL_USER_AUTH_ENABLED, syncGroupMembershipsAfterAuth.getValue());

        pollerConfig.copyTo(directory.getAttributes());
    }


    protected long getAttributeValueAsLong(Directory directory, String attributeName) {
        String value = directory.getValue(attributeName);
        return value == null ? 0 : Long.parseLong(value);
    }

    public boolean isIncrementalSyncEnabled() {
        return incrementalSyncEnabled;
    }

    public void setIncrementalSyncEnabled(boolean incrementalSyncEnabled) {
        this.incrementalSyncEnabled = incrementalSyncEnabled;
    }

    public long getPollingIntervalInMin() {
        return pollerConfig.getPollingIntervalInMin();
    }

    public void setPollingIntervalInMin(long pollingIntervalInMin) {
        pollerConfig.setPollingIntervalInMin(pollingIntervalInMin);
    }

    public String getSyncGroupMembershipsAfterAuth() {
        return syncGroupMembershipsAfterAuth.name();
    }

    public void setSyncGroupMembershipsAfterAuth(String syncGroupMembershipsAfterAuth) {
        this.syncGroupMembershipsAfterAuth = SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.valueOf(syncGroupMembershipsAfterAuth);
    }

    public String getCronExpression() {
        return pollerConfig.getCronExpression();
    }

    public void setCronExpression(String cronExpression) {
        pollerConfig.setCronExpression(cronExpression);
    }

    public String getSynchronisationType() {
        return pollerConfig.getSynchronisationType();
    }

    public void setSynchronisationType(String synchronisationType) {
        pollerConfig.setSynchronisationType(synchronisationType);
    }

    public Map<String, String> getSyncGroupMembershipsAfterAuthValues() {
        return SyncGroupMembershipsUtils.syncGroupMembershipsAfterAuthOptions();
    }

    public String getServerTimeZone() {
        return serverTimeZone;
    }
}
