package com.atlassian.crowd.directory.ldap.util;

import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;

/**
 * Util for  GUID (externalId) transformations
 *
 * @since v2.7.0
 */
public class GuidHelper {

    public static final String BS = "\\";

    /**
     * The returned representation doesn't match AD's string representation,
     * but it doesn't matter as the GUID should be treated as an opaque
     * identifier. Basically, the method is a byte array to hex string.
     * AD chooses to order the hex string in partial reverse, eg.
     *
     * Normal Hex String:     6797e1e5ecb5154f960f865c28c015fa
     * AD Formatted String:   e5e19767-b5ec-4f15-960f-865c28c015fa
     *
     * This method returns the "normal" hex string.
     *
     * @param inArr guid
     * @return guid in byte-string form
     */
    public static String getGUIDAsString(byte[] inArr) {
        StringBuffer guid = new StringBuffer();
        for (int i = 0; i < inArr.length; i++) {
            StringBuffer dblByte = new StringBuffer(Integer.toHexString(inArr[i] & 0xff));
            if (dblByte.length() == 1) {
                guid.append("0");
            }
            guid.append(dblByte);
        }
        return guid.toString();
    }

    /**
     * Returns Byte String representation of GUID suitable for search.
     * <p>Example:
     * <pre>
     * Input:  6797e1e5ecb5154f960f865c28c015fa
     * Result: \67\97\e1\e5\ec\b5\15\4f\96\0f\86\5c\28\c0\15\fa
     * </pre>
     *
     * @param guid the guid to be used in search
     * @return byte string encoded form for AD search
     */
    public static String encodeGUIDForSearch(String guid) {
        Preconditions.checkArgument(guid.length() == 32, "guid should be of length 32 (as encoded by getGUIDAsString)");
        StringBuilder sb = new StringBuilder(48);
        sb.append(BS);

        Iterable<String> split = Splitter.fixedLength(2).split(guid);
        return Joiner.on(BS).appendTo(sb, split).toString();
    }

}
