package com.atlassian.crowd.directory.ldap.mapper;

import java.util.Set;

import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.atlassian.crowd.directory.ldap.util.DirectoryAttributeRetriever;

import com.google.common.collect.ImmutableSet;

import org.springframework.ldap.UncategorizedLdapException;
import org.springframework.ldap.core.DirContextAdapter;

/**
 * Maps the externalId attribute. ExternalId attribute name is read from supplied properties mapper.
 *
 * @see com.atlassian.crowd.directory.ldap.util.DirectoryAttributeRetriever#getValueFromExternalIdAttribute(String, javax.naming.directory.Attributes)
 */
public class ExternalIdContextMapper implements ContextMapperWithRequiredAttributes<String> {
    private final LDAPPropertiesMapper ldapPropertiesMapper;

    public ExternalIdContextMapper(LDAPPropertiesMapper ldapPropertiesMapper) {
        this.ldapPropertiesMapper = ldapPropertiesMapper;
    }

    /**
     * Returns objectGUID.
     *
     * @return objectGUID
     */
    @Override
    public String mapFromContext(Object ctx) {
        try {
            DirContextAdapter context = (DirContextAdapter) ctx;
            return DirectoryAttributeRetriever.getValueFromExternalIdAttribute(ldapPropertiesMapper.getExternalIdAttribute(), context.getAttributes());
        } catch (UncategorizedLdapException e) {
            throw new RuntimeException("Could not retrieve externalId from object: " + ctx, e);
        }
    }

    @Override
    public Set<String> getRequiredLdapAttributes() {
        return ImmutableSet.of(ldapPropertiesMapper.getExternalIdAttribute());
    }
}
