package com.atlassian.crowd.directory.ldap.credential;

import com.atlassian.crowd.directory.ldap.credential.LDAPCredentialEncoder.LDAPCredentialToStringEncoder;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.password.encoder.PasswordEncoder;
import com.atlassian.crowd.password.factory.PasswordEncoderFactory;

import org.apache.commons.lang3.StringUtils;

/**
 * Translates a clear-text password into an encrypted one if it isn't already encrypted, using the encryption method
 * specified by the directory settings.
 */
public class EncryptingCredentialEncoder implements LDAPCredentialToStringEncoder {
    private final PasswordEncoderFactory passwordEncoderFactory;
    private final String encryptionAlgorithm;

    public EncryptingCredentialEncoder(PasswordEncoderFactory passwordEncoderFactory, String encryptionAlgorithm) {
        this.passwordEncoderFactory = passwordEncoderFactory;
        this.encryptionAlgorithm = encryptionAlgorithm;
    }

    /**
     * @throws com.atlassian.crowd.exception.PasswordEncoderNotFoundException (runtime exception) if the encoder for the
     *                                                                        specified encryption method cannot be found
     */
    @Override
    public String encodeCredential(PasswordCredential passwordCredential) throws InvalidCredentialException {
        if (passwordCredential.isEncryptedCredential() || StringUtils.isBlank(encryptionAlgorithm)) {
            return passwordCredential.getCredential();
        } else {
            PasswordEncoder passwordEncoder = passwordEncoderFactory.getLdapEncoder(encryptionAlgorithm);
            return passwordEncoder.encodePassword(passwordCredential.getCredential(), null);
        }
    }

    @Override
    public boolean supportsSettingEncryptedPasswords() {
        return true;
    }

    public PasswordEncoderFactory getPasswordEncoderFactory() {
        return passwordEncoderFactory;
    }

    public String getEncryptionAlgorithm() {
        return encryptionAlgorithm;
    }
}
