package com.atlassian.crowd.directory.ldap.credential;

import java.io.UnsupportedEncodingException;

import com.atlassian.crowd.directory.ldap.credential.LDAPCredentialEncoder.LDAPCredentialToByteArrayEncoder;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.InvalidCredentialException;

/**
 * An encoder specifically for Microsoft Active Directory that first delegates to the provided base encoder and then
 * converts the result of that to the byte[] format that Active Directory expects.
 */
public class ActiveDirectoryCredentialEncoder implements LDAPCredentialToByteArrayEncoder {
    private static final String AD_PASSWORD_ENCODED = "UTF-16LE";

    private final LDAPCredentialToStringEncoder baseEncoder;

    public ActiveDirectoryCredentialEncoder(LDAPCredentialToStringEncoder baseEncoder) {
        this.baseEncoder = baseEncoder;
    }

    @Override
    public byte[] encodeCredential(PasswordCredential passwordCredential) throws InvalidCredentialException {
        String guaranteedUnencryptedPassword = baseEncoder.encodeCredential(passwordCredential);

        return encodeValueForUnicodePwdAttr(guaranteedUnencryptedPassword);
    }

    /**
     * Converts the clear-text password to the {@link #AD_PASSWORD_ENCODED} encoding.
     *
     * @param unhashedPasswordToEncode raw password (active directory doesn't support setting hashed passwords)
     * @return the password transformed to a format which is suitable for being set as the value of the unicodePwd attr
     * @throws InvalidCredentialException if the encoding {@value #AD_PASSWORD_ENCODED} does not exist on this system.
     */
    private static byte[] encodeValueForUnicodePwdAttr(String unhashedPasswordToEncode) throws InvalidCredentialException {
        try {
            //Replace the "unicodePwd" attribute with a new value
            //Password must be both Unicode and a quoted string
            String newQuotedPassword = "\"" + unhashedPasswordToEncode + "\"";
            return newQuotedPassword.getBytes(AD_PASSWORD_ENCODED);
        } catch (UnsupportedEncodingException e)  // if the Charset AD_PASSWORD_ENCODED isn't available on this system
        {
            throw new InvalidCredentialException(e.getMessage(), e);
        }
    }

    @Override
    public boolean supportsSettingEncryptedPasswords() {
        return baseEncoder.supportsSettingEncryptedPasswords();
    }

    public LDAPCredentialToStringEncoder getBaseEncoder() {
        return baseEncoder;
    }
}
