package com.atlassian.crowd.directory.ldap.cache;

import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;

import com.atlassian.crowd.directory.MicrosoftActiveDirectory;
import com.atlassian.crowd.directory.RFC4519DirectoryMembershipsIterable;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.DirectoryEntities;
import com.atlassian.crowd.model.Tombstone;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.LDAPGroupWithAttributes;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.LDAPUserWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.fugue.Pair;
import com.atlassian.util.concurrent.ThreadFactories;

import com.google.common.base.Function;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.collect.Iterables.transform;

/**
 * Retrieves latest changes from MS Active Directory in order to allow "delta" cache refreshes.
 * <p>
 * See http://msdn.microsoft.com/en-us/library/ms677625.aspx and http://msdn.microsoft.com/en-us/library/ms677627%28VS.85%29.aspx
 * for details on polling Microsoft Active Directory.
 * <p>
 * This class is guaranteed to be run from a single thread at a time by per directory.  This means it does not need to
 * worry about race-conditions, but still must consider safe publication of variables (per directory).
 *
 * The incremental sync of users comes in two flavors:
 * <ul>
 * <li>Legacy mode - based on usnChanged attribute only
 * <li>New 'bulletproof' mode (default) - using both usnChanged and ObjectGUID diff between internal and remote directory
 * </ul>
 * It is possible to switch back to 'legacy mode' by supplying a system property named {@link #PROPERTY_USE_LEGACY_AD_INCREMENTAL_SYNC} set to {@code true}.
 */
public class UsnChangedCacheRefresher extends AbstractCacheRefresher implements CacheRefresher {
    private static final Logger log = LoggerFactory.getLogger(UsnChangedCacheRefresher.class);
    private static final long UNINITIALISED = -1;

    public static final Function<Tombstone, String> TOMBSTONE_TO_GUID_FUNCTION = new Function<Tombstone, String>() {
        @Override
        public String apply(Tombstone tombstone) {
            return tombstone.getObjectGUID();
        }
    };

    public static final String PROPERTY_USE_LEGACY_AD_INCREMENTAL_SYNC = "crowd.use.legacy.ad.incremental.sync";

    private final MicrosoftActiveDirectory activeDirectory;

    // the following caches should be stored in a cluster-safe manner
    // make volatile for safe-publication
    private volatile long highestCommittedUSN = UNINITIALISED;

    final LDAPEntityNameMap<LDAPUserWithAttributes> userMap = new LDAPEntityNameMap<LDAPUserWithAttributes>();
    private final LDAPEntityNameMap<LDAPGroupWithAttributes> groupMap = new LDAPEntityNameMap<LDAPGroupWithAttributes>();

    private Future<List<LDAPUserWithAttributes>> userListFuture;
    private Future<List<LDAPGroupWithAttributes>> groupListFuture;

    private final boolean useLegacyADIncrementalSync = Boolean.valueOf(System.getProperty(PROPERTY_USE_LEGACY_AD_INCREMENTAL_SYNC, "false"));

    public UsnChangedCacheRefresher(final MicrosoftActiveDirectory activeDirectory) {
        super(activeDirectory);
        this.activeDirectory = activeDirectory;
    }

    public boolean synchroniseChanges(DirectoryCache directoryCache) throws OperationFailedException {
        if (!isIncrementalSyncEnabled()) {
            return false;
        }

        // When restarting the app, we must do a full refresh the first time.
        if (highestCommittedUSN == UNINITIALISED) {
            log.info("After restarting, full sync of directory [{}] is necessary before incremental sync is possible.",
                    activeDirectory.getDirectoryId());
            return false;
        }

        // Find the latest USN-Changed value for the AD server
        long currentHighestCommittedUSN = activeDirectory.fetchHighestCommittedUSN();

        synchroniseUserChanges(directoryCache);
        synchroniseGroupChanges(directoryCache);

        // Remember the USN-Changed value that we were on when we started  (this ensures we don't miss anything, but we may get duplicates)
        this.highestCommittedUSN = currentHighestCommittedUSN;

        return true;
    }

    public void synchroniseAll(DirectoryCache directoryCache) throws OperationFailedException {
        ExecutorService queryExecutor = Executors.newFixedThreadPool(3, ThreadFactories.namedThreadFactory("CrowdUsnChangedCacheRefresher"));
        try {
            userListFuture = queryExecutor.submit(new Callable<List<LDAPUserWithAttributes>>() {
                public List<LDAPUserWithAttributes> call() throws Exception {
                    long start = System.currentTimeMillis();
                    log.debug("loading remote users");
                    List<LDAPUserWithAttributes> ldapUsers = activeDirectory.searchUsers(QueryBuilder
                            .queryFor(LDAPUserWithAttributes.class, EntityDescriptor.user())
                            .returningAtMost(EntityQuery.ALL_RESULTS));
                    log.info("found [ " + ldapUsers.size() + " ] remote users in [ " + (System.currentTimeMillis() - start) + "ms ]");
                    return ldapUsers;
                }
            });
            groupListFuture = queryExecutor.submit(new Callable<List<LDAPGroupWithAttributes>>() {
                public List<LDAPGroupWithAttributes> call() throws Exception {
                    long start = System.currentTimeMillis();
                    log.debug("loading remote groups");
                    List<LDAPGroupWithAttributes> ldapGroups = activeDirectory.searchGroups(QueryBuilder
                            .queryFor(LDAPGroupWithAttributes.class, EntityDescriptor.group(GroupType.GROUP))
                            .returningAtMost(EntityQuery.ALL_RESULTS));
                    log.info("found [ " + ldapGroups.size() + " ] remote groups in [ " + (System.currentTimeMillis() - start) + "ms ]");
                    return ldapGroups;
                }
            });

            // Find the latest USN-Changed value for the AD server
            long currentHighestCommittedUSN = activeDirectory.fetchHighestCommittedUSN();
            // Do standard synchroniseAll
            super.synchroniseAll(directoryCache);
            // Remember the USN-Changed value that we were on when we started  (this ensures we don't miss anything, but we may get duplicates)
            this.highestCommittedUSN = currentHighestCommittedUSN;
        } finally {
            queryExecutor.shutdown();

            userListFuture = null;
            groupListFuture = null;
        }
    }

    @Override
    protected List<? extends UserWithAttributes> synchroniseAllUsers(DirectoryCache directoryCache, boolean withAttributes) throws OperationFailedException {
        userMap.clear();

        Date syncStartDate = new Date();

        try {
            List<LDAPUserWithAttributes> ldapUsers = userListFuture.get();

            userMap.putAll(ldapUsers);
            //the order of this operations is relevant for rename to work properly
            directoryCache.deleteCachedUsersNotIn(ldapUsers, syncStartDate);
            directoryCache.addOrUpdateCachedUsers(ldapUsers, syncStartDate);

            return ldapUsers;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new OperationFailedException("background query interrupted", e);
        } catch (ExecutionException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    protected List<? extends GroupWithAttributes> synchroniseAllGroups(DirectoryCache directoryCache) throws OperationFailedException {
        groupMap.clear();

        Date syncStartDate = new Date();

        try {
            List<LDAPGroupWithAttributes> ldapGroups = groupListFuture.get();

            //CWD-2504: We filter out the duplicate groups to stop the runtime exception that will occur during membership
            // synchronization. There is no point registering a group we are not going to use.
            ldapGroups = Collections.unmodifiableList(DirectoryEntities.filterOutDuplicates(ldapGroups));

            groupMap.putAll(ldapGroups);

            directoryCache.deleteCachedGroupsNotIn(GroupType.GROUP, ldapGroups, syncStartDate);
            directoryCache.addOrUpdateCachedGroups(ldapGroups, syncStartDate);
            return ldapGroups;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new OperationFailedException("background query interrupted", e);
        } catch (ExecutionException e) {
            throw new OperationFailedException(e);
        }
    }

    private void synchroniseUserChangesUsn(DirectoryCache directoryCache) throws OperationFailedException {
        long start = System.currentTimeMillis();
        log.debug("loading changed remote users");
        List<LDAPUserWithAttributes> updatedUsers = activeDirectory.findAddedOrUpdatedUsersSince(highestCommittedUSN);
        List<Tombstone> tombstones = activeDirectory.findUserTombstonesSince(highestCommittedUSN);
        userMap.putAll(updatedUsers);

        log.info("found [ {} ] changed remote users in [ {}ms ]", updatedUsers.size() + tombstones.size(), (System.currentTimeMillis() - start));

        // calculate removed principals
        start = System.currentTimeMillis();
        final Set<String> tombstonesGuids = ImmutableSet.copyOf(transform(tombstones, TOMBSTONE_TO_GUID_FUNCTION));

        log.info("scanned and compared [ {} ] users for delete in DB cache in [ {}ms ]", tombstones.size(), (System.currentTimeMillis() - start));

        directoryCache.deleteCachedUsersByGuid(tombstonesGuids);
        userMap.removeAllByGuid(tombstonesGuids);

        // Send a null sync date - we want to force this change else we may miss it.
        // If we put a stale value in then it will be fixed on next refresh.
        directoryCache.addOrUpdateCachedUsers(updatedUsers, null);
    }

    /**
     * Returns user guids to be added and removed unless it detects the incremental sync is not possible in which case it throws an UsnChangedCacheRefresherIncSyncException.
     * This method intentionally does two things - it tries to fail as fast as possible. Some operations, like AD query, might be time consuming and we don't want to
     * wait unnecessarily long only to find out the incremental sync cannot be performed.
     *
     * @param directoryCache directory cache
     * @return Pair containing user guids to be added (left) and guids to be removed (right) from cache
     * @throws OperationFailedException                 when directoryCache or external directory operations throws OperationFailedException
     * @throws UsnChangedCacheRefresherIncSyncException when it detects the incremental sync should not be performed due to data inconsistency
     */
    private Pair<? extends Set<String>, ? extends Set<String>> validateAndReturnGuidsToAddAndDelete(DirectoryCache directoryCache) throws OperationFailedException {
        if (!activeDirectory.isExternalIdConfigured()) {
            throw new UsnChangedCacheRefresherIncSyncException("externalId attribute is not configured in directory.");
        }
        log.debug("loading changed users");

        // check cache first, in case any null/empty guids are found it will fail faster
        final Set<String> guidsInCache = ImmutableSet.copyOf(directoryCache.getAllUserGuids());
        if (guidsInCache.size() != directoryCache.getUserCount()) {
            throw new UsnChangedCacheRefresherIncSyncException("Cache returned different number of guids and users (possible reason is overlapping guids in cache," +
                    " most likely null/empty values).");
        }
        if (guidsInCache.contains("")) {
            throw new UsnChangedCacheRefresherIncSyncException("Empty guids returned from cache. Falling back to a full sync in order to populate the guids");
        }

        // fetch all guids from AD
        final Set<String> guidsInAD = activeDirectory.findAllUserGuids();
        // in case the externalId directory attribute is misconfigured or the AD itself is misconfigured we might get empty guids
        if (guidsInAD.contains("")) {
            throw new UsnChangedCacheRefresherIncSyncException("Empty guids returned from AD. Possible reasons are externalId attribute value in directory configuration" +
                    " or AD server configuration.");
        }

        return Pair.pair(Sets.difference(guidsInAD, guidsInCache), Sets.difference(guidsInCache, guidsInAD));
    }

    /**
     * This synchronizes users based on guids and usn.
     * Users to be added/removed are determined using diff in guids from internal and remote directory.
     * Users to be updated are calculated using usn.
     *
     * @param directoryCache directory cache
     * @throws OperationFailedException                 when directoryCache or external directory operations throws OperationFailedException
     * @throws UsnChangedCacheRefresherIncSyncException when incremental sync failed or cannot be performed due to data consistency problems
     */
    private void synchroniseUserChangesGuid(DirectoryCache directoryCache) throws OperationFailedException {
        long start = System.currentTimeMillis();

        final Pair<? extends Set<String>, ? extends Set<String>> guidsToAddAndRemove = validateAndReturnGuidsToAddAndDelete(directoryCache);
        final Set<String> guidsToAdd = guidsToAddAndRemove.left();
        final Set<String> guidsToRemove = guidsToAddAndRemove.right();

        // when queried for changes by usn the AD will return both updated and newly added users
        // we want to perform updates and additions separately in order to avoid problems (existing user X renamed to Y, new user X added)
        // therefore we filter this result
        final ImmutableMap.Builder<String, LDAPUserWithAttributes> usersToAddByGuidBuilder = ImmutableMap.builder();
        final ImmutableList.Builder<LDAPUserWithAttributes> usersToUpdateBuilder = ImmutableList.builder();
        for (LDAPUserWithAttributes user : activeDirectory.findAddedOrUpdatedUsersSince(highestCommittedUSN)) {
            final String externalId = user.getExternalId();
            // in the unlikely event of AD returning null or empty externalIds right after we checked this with the validateAndReturnGuidsToAddAndDelete()
            // we throw an exception to fallback to a full sync
            if (StringUtils.isEmpty(externalId)) {
                throw new UsnChangedCacheRefresherIncSyncException("A null or empty guid retrieved from AD.");
            }
            if (guidsToAdd.contains(externalId)) {
                // this user is already referenced by toAdd set so it needs to be filtered out.
                // in order to reduce the number of calls to AD, when we fetch users to be added from AD by guid, we store this user in cache.
                usersToAddByGuidBuilder.put(externalId, user);
            } else {
                usersToUpdateBuilder.add(user);
            }
        }
        final ImmutableMap<String, LDAPUserWithAttributes> usersToAddByGuid = usersToAddByGuidBuilder.build();
        final ImmutableList<LDAPUserWithAttributes> usersToUpdate = usersToUpdateBuilder.build();

        log.info("scanned and compared [ {} ] users to delete, [ {} ] users to add, [ {} ] users to update in DB cache in [ {}ms ]"
                , guidsToRemove.size(), guidsToAdd.size(), usersToUpdate.size(), (System.currentTimeMillis() - start));

        // 1 - delete users by guid
        directoryCache.deleteCachedUsersByGuid(guidsToRemove);
        userMap.removeAllByGuid(guidsToRemove);

        // 2 - update changed users (usn)
        // Send a null sync date - we want to force this change else we may miss it.
        // If we put a stale value in then it will be fixed on next refresh.
        directoryCache.addOrUpdateCachedUsers(usersToUpdate, null);
        userMap.putAll(usersToUpdate);

        // 3 - add new users by guid, but first fetch their User data
        List<LDAPUserWithAttributes> newUsers = Lists.newArrayList(
                transform(guidsToAdd, getGuidToUserFunction(usersToAddByGuid))
        );
        directoryCache.addOrUpdateCachedUsers(newUsers, null);
        userMap.putAll(newUsers);
    }

    void synchroniseUserChanges(DirectoryCache directoryCache) throws OperationFailedException {
        if (useLegacyADIncrementalSync) {
            synchroniseUserChangesUsn(directoryCache);
        } else {
            synchroniseUserChangesGuid(directoryCache);
        }
    }

    private void synchroniseGroupChanges(DirectoryCache directoryCache) throws OperationFailedException {
        long start = System.currentTimeMillis();
        log.debug("loading changed remote groups");
        //CWD-2504: We filter out the duplicate groups to stop the runtime exception that will occur during membership
        // synchronization.
        List<LDAPGroupWithAttributes> updatedGroups = DirectoryEntities.filterOutDuplicates(activeDirectory.findAddedOrUpdatedGroupsSince(highestCommittedUSN));
        List<Tombstone> tombstones = activeDirectory.findGroupTombstonesSince(highestCommittedUSN);
        log.info(new StringBuilder("found [ ").append(updatedGroups.size() + tombstones.size()).append(" ] changed remote groups in [ ").append(System.currentTimeMillis() - start).append("ms ]").toString());

        groupMap.putAll(updatedGroups);

        // Send a null sync date - we want to force this change else we may miss it.
        // If we put a stale value in then it will be fixed on next refresh.
        directoryCache.addOrUpdateCachedGroups(updatedGroups, null);

        // Now update the memberships for the changed groups
        // (After new Groups are added because we may have nested group memberships)
        synchroniseMemberships(updatedGroups, directoryCache);

        // calculate removed groups
        start = System.currentTimeMillis();
        Set<String> groupnames = new HashSet<String>();
        for (Tombstone tombstone : tombstones) {
            String groupName = groupMap.getByGuid(tombstone.getObjectGUID());
            if (groupName != null) {
                groupnames.add(groupName);
            }
        }
        log.info(new StringBuilder("scanned and compared [ ").append(tombstones.size()).append(" ] groups for delete in DB cache in [ ").append(System.currentTimeMillis() - start).append("ms ]").toString());

        directoryCache.deleteCachedGroups(groupnames);
    }

    @Override
    Iterable<Membership> getMemberships(Iterable<String> names) throws OperationFailedException {
        try {
            Map<LdapName, String> users = userMap.toLdapNameKeyedMap();
            Map<LdapName, String> groups = groupMap.toLdapNameKeyedMap();

            return new RFC4519DirectoryMembershipsIterable(activeDirectory, users, groups, ImmutableSet.copyOf(names));
        } catch (InvalidNameException e) {
            throw new OperationFailedException("Failed to get directory memberships due to invalid DN", e);
        }
    }

    /**
     * Returns a function to retrieve users by externalId from AD.
     *
     * @param userCache map of users by externalId, serves as a cache to avoid unnecessary calls to AD
     * @return function to map from externalId to LDAPUserWithAttributes
     */
    private Function<String, LDAPUserWithAttributes> getGuidToUserFunction(final Map<String, LDAPUserWithAttributes> userCache) {
        return new GuidToUserFunction(activeDirectory, userCache);
    }

    /**
     * Function to retrieve users by externalId from AD. This function attempts to reduce the number of
     * calls to AD by trying to lookup the user in supplied map.
     * This function will throw a RuntimeException when user could not be found in both cache and AD
     */
    static class GuidToUserFunction implements Function<String, LDAPUserWithAttributes> {
        private final MicrosoftActiveDirectory activeDirectory;
        private final Map<String, LDAPUserWithAttributes> userCache;

        /**
         * Constructor.
         *
         * @param activeDirectory Acitve Directory connector
         * @param userCache       map of users by externalId to be used instead of querying the AD
         */
        GuidToUserFunction(MicrosoftActiveDirectory activeDirectory, Map<String, LDAPUserWithAttributes> userCache) {
            this.activeDirectory = activeDirectory;
            this.userCache = userCache;
        }

        @Override
        public LDAPUserWithAttributes apply(String externalId) {
            try {
                LDAPUserWithAttributes userFromCache = userCache.get(externalId);
                if (userFromCache != null) {
                    return userFromCache;
                } else {
                    return activeDirectory.findUserByExternalId(externalId);
                }
            } catch (UserNotFoundException e) {
                log.warn("User with objectGUID '{}' not found in ActiveDirectory", externalId);
            } catch (OperationFailedException e) {
                log.warn("Failed to fetch user by objectGUID '{}' from ActiveDirectory", externalId, e);
            }
            // the only reasonable thing is to fallback to a full sync
            throw new UsnChangedCacheRefresherIncSyncException("Problems while looking up users by objectGUID in ActiveDirectory detected, falling back to a full sync.");
        }
    }
}
