package com.atlassian.crowd.directory.ldap.cache;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.DirectoryEntities;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A simple implementation of CacheRefresher that will only do "Full Refresh".
 * This is used for all LDAP servers other than AD.
 *
 * @since v2.1
 */
public class RemoteDirectoryCacheRefresher extends AbstractCacheRefresher implements CacheRefresher {
    private static final Logger log = LoggerFactory.getLogger(RemoteDirectoryCacheRefresher.class);
    public static final EntityQuery<GroupWithAttributes> ALL_REMOTE_GROUPS_QUERY = QueryBuilder.queryFor(GroupWithAttributes.class, EntityDescriptor.group(GroupType.GROUP)).returningAtMost(EntityQuery.ALL_RESULTS);

    public RemoteDirectoryCacheRefresher(final RemoteDirectory remoteDirectory) {
        super(remoteDirectory);
    }

    public boolean synchroniseChanges(final DirectoryCache directoryCache) throws OperationFailedException {
        // We can never do a delta sync
        return false;
    }

    private List<UserWithAttributes> findAllRemoteUsers(boolean withAttributes) throws OperationFailedException {
        long start = System.currentTimeMillis();
        log.debug("loading remote users");
        List<UserWithAttributes> users;
        if (withAttributes) {
            users = remoteDirectory.searchUsers(getUserQuery(UserWithAttributes.class));
        } else {
            users = remoteDirectory.searchUsers(getUserQuery(User.class)).stream()
                    .map(UserTemplateWithAttributes::toUserWithNoAttributes)
                    .collect(Collectors.toList());
        }
        log.info("found [ {} ] remote users in [ {} ms ]", users.size(), (System.currentTimeMillis() - start));
        return users;
    }

    private <T extends User> EntityQuery<T> getUserQuery(Class<T> clazz) {
        return QueryBuilder.queryFor(clazz, EntityDescriptor.user())
                .returningAtMost(EntityQuery.ALL_RESULTS);
    }

    private List<GroupWithAttributes> findAllRemoteGroups() throws OperationFailedException {
        long start = System.currentTimeMillis();
        log.debug("loading remote groups");
        List<GroupWithAttributes> groups = remoteDirectory.searchGroups(ALL_REMOTE_GROUPS_QUERY);
        log.info("found [ {} ] remote groups in [ {} ms ]", groups.size(), (System.currentTimeMillis() - start));
        return groups;
    }

    @Override
    protected List<? extends UserWithAttributes> synchroniseAllUsers(DirectoryCache directoryCache, boolean withAttributes) throws OperationFailedException {
        Date syncStartDate = new Date();

        List<? extends UserWithAttributes> ldapUsers = findAllRemoteUsers(withAttributes);

        // By removing missing users before adding/updating, users that have been renamed can be tracked correctly.
        // See JDEV-22181 for a more in-depth discussion.
        directoryCache.deleteCachedUsersNotIn(ldapUsers, syncStartDate);
        directoryCache.addOrUpdateCachedUsers(ldapUsers, syncStartDate);

        return ldapUsers;
    }

    @Override
    protected List<? extends GroupWithAttributes> synchroniseAllGroups(DirectoryCache directoryCache) throws OperationFailedException {
        Date syncStartDate = new Date();

        //CWD-2504: We filter out the duplicate groups. This basically means that these groups don't exist to crowd.
        // Crowd can't currently deal with groups with the same name so rather than throwing a runtime exception
        // which kills the synchronization, we just ignore them and log a message and continue to work.
        // The admins of the LDAP directory will have to do some admin to get the groups to work correctly.
        List<? extends GroupWithAttributes> groups = DirectoryEntities.filterOutDuplicates(findAllRemoteGroups());

        // By removing missing groups before adding/updating, groups that have been renamed in a way that looks the same
        // to the database will be updated in one sync instead of needing two.
        // See CWD-4290 for the motivating example.
        directoryCache.deleteCachedGroupsNotIn(GroupType.GROUP, groups, syncStartDate);
        directoryCache.addOrUpdateCachedGroups(groups, syncStartDate);

        return groups;
    }
}
