package com.atlassian.crowd.directory.ldap.cache;

import java.util.Collection;
import java.util.Map;

import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;

import com.atlassian.crowd.directory.ldap.mapper.attribute.ObjectGUIDMapper;
import com.atlassian.crowd.model.LDAPDirectoryEntity;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import com.google.common.collect.Maps;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

/**
 * A composite map of ObjectGUID to DN and DN to Name.
 * Completely not thread-safe.
 *
 * @param <T> element type group or user
 */
public class LDAPEntityNameMap<T extends LDAPDirectoryEntity> {
    @VisibleForTesting
    final Map<String, String> guidMap = Maps.newHashMap();
    @VisibleForTesting
    final Map<String, String> dnMap = Maps.newHashMap();
    /**
     * This map has been introduced to avoid linear search in dnMap's values.
     */
    @VisibleForTesting
    final Map<String, String> guidToDnMap = Maps.newHashMap();

    public void putAll(Collection<T> entities) {
        for (T ldapEntity : entities) {
            put(ldapEntity);
        }
    }

    public void put(T ldapEntity) {
        final String objectGUID = ldapEntity.getValue(ObjectGUIDMapper.ATTRIBUTE_KEY);
        // maintain integrity - when an entity is added twice, the second time with dn/name changed (entity is renamed, dn is different but objectGUID remains the same)
        final String dn = guidToDnMap.get(objectGUID);
        if (dn != null) {
            dnMap.remove(dn);
        }
        guidToDnMap.put(objectGUID, ldapEntity.getDn());

        guidMap.put(objectGUID, ldapEntity.getName());
        dnMap.put(ldapEntity.getDn(), ldapEntity.getName());
    }

    public void removeByGuid(String guid) {
        guidMap.remove(guid);
        final String dn = guidToDnMap.get(guid);
        dnMap.remove(dn);
        guidToDnMap.remove(guid);
    }

    public void removeAllByGuid(Collection<String> guids) {
        for (String guid : guids) {
            removeByGuid(guid);
        }
    }

    public String getByDn(String dn) {
        return dnMap.get(dn);
    }

    public String getByGuid(String guid) {
        return guidMap.get(guid);
    }

    public void clear() {
        guidMap.clear();
        dnMap.clear();
        guidToDnMap.clear();
    }

    @SuppressFBWarnings(value = "LDAP_INJECTION", justification = "No user input")
    public Map<LdapName, String> toLdapNameKeyedMap() throws InvalidNameException {
        Builder<LdapName, String> builder = ImmutableMap.<LdapName, String>builder();

        for (Map.Entry<String, String> e : dnMap.entrySet()) {
            builder.put(new LdapName(e.getKey()), e.getValue());
        }

        return builder.build();
    }
}
