package com.atlassian.crowd.directory;

import javax.naming.directory.Attributes;

import com.atlassian.crowd.directory.ldap.credential.EnforceUnencryptedCredentialEncoder;
import com.atlassian.crowd.directory.ldap.credential.LDAPCredentialEncoder;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.ldap.LDAPQueryTranslater;
import com.atlassian.crowd.util.InstanceFactory;
import com.atlassian.crowd.util.PasswordHelper;
import com.atlassian.event.api.EventPublisher;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.ldap.core.LdapTemplate;
import org.springframework.ldap.core.support.LdapContextSource;

/**
 * Sun ONE / Sun DSEE Directory connector.
 */
public class SunONE extends RFC4519Directory {
    private static final Logger logger = LoggerFactory.getLogger(SunONE.class);

    private final LDAPCredentialEncoder credentialEncoder;

    /**
     * @param passwordHelper password helper, which must not be null
     */
    public SunONE(LDAPQueryTranslater ldapQueryTranslater, EventPublisher eventPublisher, InstanceFactory instanceFactory,
                  PasswordHelper passwordHelper) {
        super(ldapQueryTranslater, eventPublisher, instanceFactory);
        // Sun DSEE doesn't want passwords encoded before they're passed to the directory.
        credentialEncoder = new EnforceUnencryptedCredentialEncoder(passwordHelper);
    }

    public static String getStaticDirectoryType() {
        return "Sun Directory Server Enterprise Edition";
    }

    @Override
    public String getDescriptiveName() {
        return SunONE.getStaticDirectoryType();
    }

    @Override
    protected LDAPCredentialEncoder getCredentialEncoder() {
        return credentialEncoder;
    }

    /**
     * Sun DSEE 6.2 in a default install requires the sn to be set before a user can be created.
     */
    @Override
    protected void getNewUserDirectorySpecificAttributes(final User user, final Attributes attributes) {
        addDefaultSnToUserAttributes(attributes, "");
    }


    // CHANGE LISTENER STUFF

    protected LdapTemplate createChangeListenerTemplate() {
        // create a spring connection context object
        LdapContextSource contextSource = new LdapContextSource();

        contextSource.setUrl(ldapPropertiesMapper.getConnectionURL());
        contextSource.setUserDn(ldapPropertiesMapper.getUsername());
        contextSource.setPassword(ldapPropertiesMapper.getPassword());

        // let spring know of our connection attributes
        contextSource.setBaseEnvironmentProperties(getBaseEnvironmentProperties());

        // create a pool for when doing multiple calls.
        contextSource.setPooled(true);

        // by default, all results are converted to a DirContextAdapter using the
        // dirObjectFactory property of the ContextSource (we need to disable that)
        contextSource.setDirObjectFactory(null);

        try {
            // we need to tell the context source to configure up our ldap server
            contextSource.afterPropertiesSet();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }

        return new LdapTemplate(contextSource);
    }
}