package com.atlassian.crowd.directory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import javax.naming.Context;
import javax.naming.InvalidNameException;
import javax.naming.Name;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchControls;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.LdapName;

import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapper;
import com.atlassian.crowd.directory.ldap.LDAPPropertiesMapperImpl;
import com.atlassian.crowd.directory.ldap.SpringLdapTemplateWrapper;
import com.atlassian.crowd.directory.ldap.credential.LDAPCredentialEncoder;
import com.atlassian.crowd.directory.ldap.mapper.AttributeToContextCallbackHandler;
import com.atlassian.crowd.directory.ldap.mapper.ContextMapperWithRequiredAttributes;
import com.atlassian.crowd.directory.ldap.mapper.GroupContextMapper;
import com.atlassian.crowd.directory.ldap.mapper.JpegPhotoContextMapper;
import com.atlassian.crowd.directory.ldap.mapper.UserContextMapper;
import com.atlassian.crowd.directory.ldap.mapper.attribute.AttributeMapper;
import com.atlassian.crowd.directory.ldap.mapper.entity.LDAPGroupAttributesMapper;
import com.atlassian.crowd.directory.ldap.mapper.entity.LDAPUserAttributesMapper;
import com.atlassian.crowd.directory.ldap.name.Converter;
import com.atlassian.crowd.directory.ldap.name.GenericConverter;
import com.atlassian.crowd.directory.ldap.name.SearchDN;
import com.atlassian.crowd.directory.ldap.util.DNStandardiser;
import com.atlassian.crowd.directory.ldap.util.DirectoryAttributeRetriever;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.OperationNotSupportedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.avatar.AvatarReference;
import com.atlassian.crowd.model.LDAPDirectoryEntity;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.LDAPGroupWithAttributes;
import com.atlassian.crowd.model.user.LDAPUserWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.ldap.LDAPQuery;
import com.atlassian.crowd.search.ldap.LDAPQueryTranslater;
import com.atlassian.crowd.search.ldap.NullResultException;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.GroupQuery;
import com.atlassian.crowd.search.query.entity.restriction.NullRestrictionImpl;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.util.BoundedCount;
import com.atlassian.crowd.util.InstanceFactory;
import com.atlassian.crowd.util.UserUtils;
import com.atlassian.event.api.EventPublisher;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.ldap.NameNotFoundException;
import org.springframework.ldap.NamingException;
import org.springframework.ldap.control.PagedResultsDirContextProcessor;
import org.springframework.ldap.core.CollectingNameClassPairCallbackHandler;
import org.springframework.ldap.core.ContextMapper;
import org.springframework.ldap.core.ContextSource;
import org.springframework.ldap.core.DirContextProcessor;
import org.springframework.ldap.core.LdapTemplate;
import org.springframework.ldap.core.support.AggregateDirContextProcessor;
import org.springframework.ldap.core.support.LdapContextSource;
import org.springframework.ldap.transaction.compensating.manager.ContextSourceTransactionManager;
import org.springframework.ldap.transaction.compensating.manager.TransactionAwareContextSourceProxy;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionException;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

import static com.atlassian.crowd.search.util.SearchResultsUtil.constrainResults;
import static com.atlassian.crowd.search.util.SearchResultsUtil.convertEntitiesToNames;
import static com.google.common.base.MoreObjects.firstNonNull;

/**
 * This class implements a remote LDAP directory using Spring LdapTemplate.
 * <p>
 * Warning: CWD-2494: When read timeout is enabled, operations can fail
 * randomly with "javax.naming.NamingException: LDAP response read timed out..."
 * error message without waiting for the timeout to pass.
 */
public abstract class SpringLDAPConnector implements LDAPDirectory {
    @Override
    public BoundedCount countDirectMembersOfGroup(String groupName, int querySizeHint) throws OperationFailedException {
        final MembershipQuery<String> membershipQuery = QueryBuilder
                .queryFor(String.class, EntityDescriptor.user())
                .childrenOf(EntityDescriptor.group())
                .withName(groupName)
                .startingAt(0)
                .returningAtMost(querySizeHint);

        return BoundedCount.fromCountedItemsAndLimit(searchGroupRelationships(membershipQuery).size(), querySizeHint);
    }

    public static final int DEFAULT_PAGE_SIZE = 999;

    private static final Logger logger = LoggerFactory.getLogger(SpringLDAPConnector.class);

    // configuration parameters (initialisation)
    private volatile long directoryId;
    protected volatile AttributeValuesHolder attributes;

    // derived configuration
    protected volatile SpringLdapTemplateWrapper ldapTemplate;
    protected volatile ContextSource contextSource;
    protected volatile Converter nameConverter;
    protected volatile SearchDN searchDN;
    protected volatile LDAPPropertiesMapper ldapPropertiesMapper;
    protected volatile ContextSourceTransactionManager contextSourceTransactionManager;

    // spring injected dependencies
    protected final LDAPQueryTranslater ldapQueryTranslater;
    protected final EventPublisher eventPublisher;
    private final InstanceFactory instanceFactory;

    public SpringLDAPConnector(LDAPQueryTranslater ldapQueryTranslater, EventPublisher eventPublisher, InstanceFactory instanceFactory) {
        this.ldapQueryTranslater = ldapQueryTranslater;
        this.eventPublisher = eventPublisher;
        this.instanceFactory = instanceFactory;
    }

    @Override
    public long getDirectoryId() {
        return this.directoryId;
    }

    /**
     * Called by the {@link com.atlassian.crowd.directory.loader.DirectoryInstanceLoader} after
     * constructing an InternalDirectory.
     *
     * @param id The unique <code>id</code> of the Directory stored in the database.
     */
    @Override
    public void setDirectoryId(final long id) {
        this.directoryId = id;
    }

    /**
     * Called by the {@link com.atlassian.crowd.directory.loader.DirectoryInstanceLoader} after
     * constructing an InternalDirectory.
     *
     * @param attributes attributes map.
     */
    @Override
    public void setAttributes(final Map<String, String> attributes) {
        this.attributes = new AttributeValuesHolder(attributes);

        // configure our LDAP helper - now getting this via Spring
        ldapPropertiesMapper = instanceFactory.getInstance(LDAPPropertiesMapperImpl.class);
        ldapPropertiesMapper.setAttributes(attributes);

        // create a spring connection context object
        contextSource = createContextSource(ldapPropertiesMapper, getBaseEnvironmentProperties());
        contextSourceTransactionManager = new ContextSourceTransactionManager();
        contextSourceTransactionManager.setContextSource(contextSource);

        ldapTemplate = new SpringLdapTemplateWrapper(new LdapTemplate(contextSource));

        // Ignore PartialResultExceptions when not following referrals
        if (!ldapPropertiesMapper.isReferral()) {
            ldapTemplate.setIgnorePartialResultException(true);
        }

        nameConverter = new GenericConverter();
        searchDN = new SearchDN(ldapPropertiesMapper, nameConverter);
    }

    private static ContextSource createContextSource(LDAPPropertiesMapper ldapPropertiesMapper, Map<String, Object> envProperties) {
        LdapContextSource targetContextSource = new LdapContextSource();

        //Attempt to look up and use the context factory if specified
        String initialContextFactoryClassName = (String) envProperties.get(Context.INITIAL_CONTEXT_FACTORY);
        if (initialContextFactoryClassName != null) {
            try {
                targetContextSource.setContextFactory(Class.forName(initialContextFactoryClassName, false, SpringLDAPConnector.class.getClassLoader()));
            } catch (ClassNotFoundException e) {
                NoClassDefFoundError err = new NoClassDefFoundError(initialContextFactoryClassName);
                err.initCause(e);
                throw err;
            }
        }

        targetContextSource.setUrl(ldapPropertiesMapper.getConnectionURL());
        targetContextSource.setUserDn(ldapPropertiesMapper.getUsername());
        targetContextSource.setPassword(ldapPropertiesMapper.getPassword());

        // let spring know of our connection attributes
        targetContextSource.setBaseEnvironmentProperties(envProperties);

        // create a pool for when doing multiple calls.
        targetContextSource.setPooled(true);

        try {
            // we need to tell the context source to configure up our ldap server
            targetContextSource.afterPropertiesSet();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }

        return new TransactionAwareContextSourceProxy(targetContextSource);
    }

    /**
     * Exposed so that delegated directories can get a handle on the underlying LDAP context.
     *
     * @return ContextSource.
     */
    public ContextSource getContextSource() {
        return contextSource;
    }

    public LDAPPropertiesMapper getLdapPropertiesMapper() {
        return ldapPropertiesMapper;
    }

    @Override
    public Set<String> getValues(final String name) {
        return attributes.getValues(name);
    }

    @Override
    public String getValue(final String name) {
        return attributes.getValue(name);
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    public long getAttributeAsLong(final String name, long defaultValue) {
        return attributes.getAttributeAsLong(name, defaultValue);
    }

    public boolean getAttributeAsBoolean(final String name, boolean defaultValue) {
        return attributes.getAttributeAsBoolean(name, defaultValue);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.getKeys();
    }

    public SearchDN getSearchDN() {
        return searchDN;
    }

    private final SearchControls getSubTreeSearchControls() {
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(SearchControls.SUBTREE_SCOPE);

        // should never ever be set to true, see CWD-4754 for more details
        searchControls.setReturningObjFlag(false);

        return searchControls;
    }

    private final SearchControls getSubTreeSearchControls(String[] returningAttributes) {
        SearchControls sc = getSubTreeSearchControls();
        if (returningAttributes != null) {
            sc.setReturningAttributes(returningAttributes);
        }
        return sc;
    }

    static final SearchControls copyOf(SearchControls sc) {
        return new SearchControls(
                sc.getSearchScope(), sc.getCountLimit(), sc.getTimeLimit(),
                sc.getReturningAttributes(), sc.getReturningObjFlag(),
                sc.getDerefLinkFlag());
    }

    private static final String[] toArray(Collection<String> s) {
        if (s != null) {
            return s.toArray(new String[s.size()]);
        } else {
            return null;
        }
    }

    protected SearchControls getSubTreeSearchControls(ContextMapperWithRequiredAttributes<?> mapper) {
        return getSubTreeSearchControls(toArray(mapper.getRequiredLdapAttributes()));
    }

    /**
     * Returns the properties used to set up the Ldap ContextSource.
     *
     * @return the properties used to set up the Ldap ContextSource.
     */
    protected Map<String, Object> getBaseEnvironmentProperties() {
        return ldapPropertiesMapper.getEnvironment();
    }

    /**
     * Performs a paged results search on an LDAP directory server searching using the LDAP paged results control
     * option to fetch results in chunks rather than all at once.
     *
     * @param baseDN              The DN to beging the search from.
     * @param filter              The search filter.
     * @param contextMapper       Maps from LDAP search results into objects such as <code>Group</code>s.
     * @param searchControls      The LDAP search scope type.
     * @param ldapRequestControls Any LDAP request controls (set to <code>null</code> if you do not need <b>additional</b> request controls for the search).
     * @param maxResults          maximum number of results to return. Set to <code>-1</code> if no result limiting is desired (WARNING: doing so is obviously a hazard).
     * @return The search results.
     * @throws OperationFailedException Search failed due to a communication error to the remote directory
     */
    protected CollectingNameClassPairCallbackHandler pageSearchResults(Name baseDN, String filter, ContextMapper contextMapper, SearchControls searchControls, DirContextProcessor ldapRequestControls, int maxResults)
            throws OperationFailedException {
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition(TransactionDefinition.PROPAGATION_REQUIRED);
        try {
            // Use a transaction in order to use the same connection for multiple LDAP requests
            TransactionStatus status = contextSourceTransactionManager.getTransaction(transactionDefinition);
            try {
                int pagingSize = ldapPropertiesMapper.getPagedResultsSize();

                PagedResultsDirContextProcessor pagedResultsControl = new PagedResultsDirContextProcessor(pagingSize);

                if (logger.isDebugEnabled()) {
                    logger.debug("Paged results are enabled with a paging size of: " + pagingSize);
                }

                // specify that we are using an object that will callback to the server doing chunks of processing users
                AttributeToContextCallbackHandler handler = new AttributeToContextCallbackHandler(contextMapper);

                // the cookie to use when maintaining our index location
                byte[] cookie = null;

                do {
                    // setup the LDAP request control(s)
                    AggregateDirContextProcessor aggregateDirContextProcessor = new AggregateDirContextProcessor();
                    aggregateDirContextProcessor.addDirContextProcessor(pagedResultsControl);
                    if (ldapRequestControls != null) {
                        aggregateDirContextProcessor.addDirContextProcessor(ldapRequestControls);
                    }

                    // perform the search
                    ldapTemplate.search(baseDN, filter, searchControls, handler, aggregateDirContextProcessor);

                    if (logger.isDebugEnabled()) {
                        int resultSize = pagedResultsControl.getPageSize();

                        logger.debug("Iterating a search result size of: " + resultSize);
                    }

                    // get the marker for the paged results list
                    pagedResultsControl = new PagedResultsDirContextProcessor(pagingSize, pagedResultsControl.getCookie());

                    // set our index pointer to the item we are currently at on the list
                    // if it is not null
                    if (pagedResultsControl.getCookie() != null) {
                        cookie = pagedResultsControl.getCookie().getCookie();
                    }
                }
                // while there are more elements keep looping AND if we don't have maxResults
                while ((cookie != null) && (cookie.length != 0) && (handler.getList().size() < maxResults || maxResults == EntityQuery.ALL_RESULTS));

                // return the results
                return handler;
            } finally {
                contextSourceTransactionManager.commit(status);
            }
        } catch (TransactionException e) {
            throw new OperationFailedException(e);
        } catch (NamingException e) {
            throw new OperationFailedException(e);
        }
    }

    /**
     * Executes a search with paging if paged results is supported.
     *
     * @param baseDN        base DN of search.
     * @param filter        encoded LDAP search filter.
     * @param contextMapper directory context to object mapper.
     * @param startIndex    index to start at. Set to <code>0</code> to start from the first result.
     * @param maxResults    maximum number of results to return. Set to <code>-1</code> if no result limiting is desired (WARNING: doing so is obviously a hazard).
     * @return list of entities of type corresponding to the contextMapper's output.
     * @throws OperationFailedException a Communication error occurred when trying to talk to a remote directory
     */
    protected <T> List<T> searchEntities(final Name baseDN, final String filter, final ContextMapperWithRequiredAttributes<T> contextMapper, final int startIndex, final int maxResults)
            throws OperationFailedException {
        return searchEntitiesWithRequestControls(baseDN, filter, contextMapper, getSubTreeSearchControls(contextMapper), null, startIndex, maxResults);
    }

    @SuppressWarnings("unchecked")
    @SuppressFBWarnings(value = "LDAP_INJECTION", justification = "No user input, filter is encoded in all calls")
    protected <T> List<T> searchEntitiesWithRequestControls(final Name baseDN, final String filter, final ContextMapperWithRequiredAttributes<T> contextMapper, SearchControls searchControls, final DirContextProcessor ldapRequestControls, final int startIndex, final int maxResults)
            throws OperationFailedException {
        List<T> results;

        searchControls = copyOf(searchControls);

        // Set the time limit for the search (if not specified in properties, will be default of 0 - no time limit)
        searchControls.setTimeLimit(ldapPropertiesMapper.getSearchTimeLimit());

        // if the directory supports paged results, use them
        if (ldapPropertiesMapper.isPagedResultsControl()) {
            CollectingNameClassPairCallbackHandler handler = pageSearchResults(baseDN, filter, contextMapper, searchControls, ldapRequestControls, startIndex + maxResults);

            results = handler.getList();
        } else {
            try {
                // otherwise fetch all the results at once

                DirContextProcessor processor = firstNonNull(ldapRequestControls, DO_NOTHING_DIR_CONTEXT_PROCESSOR);

                if (maxResults != EntityQuery.ALL_RESULTS) {
                    /* If the request is not for all results then limit the search */
                    int limit = startIndex + maxResults;

                    if (searchControls.getCountLimit() == 0) {
                        searchControls.setCountLimit(limit);
                    }

                    results = ldapTemplate.searchWithLimitedResults(baseDN, filter, searchControls, contextMapper, processor, limit);
                } else {
                    results = ldapTemplate.search(baseDN, filter, searchControls, contextMapper, processor);
                }
            } catch (NamingException ex) {
                throw new OperationFailedException(ex);
            }
        }

        if (contextMapper instanceof GroupContextMapper) {
            // Need to postprocessGroups here (for Microsoft AD) in case a group has a large number of members
            // See: https://studio.atlassian.com/browse/EMBCWD-622
            results = (List<T>) postprocessGroups((List<LDAPGroupWithAttributes>) results);
        }

        return constrainResults(results, startIndex, maxResults);
    }

    /**
     * This method is not suitable for generic attribute updates as it only supports single
     * attribute-value mappings (ie. suitable for field values as opposed to custom attributes).
     *
     * @param directoryAttributeName the name of the attribute in LDAP to potentially add or modify.
     * @param oldValue               the value load from the LDAP directory (i.e. already processed by
     *                               {@link DirectoryAttributeRetriever#fromSavedLDAPValue(String)})
     * @param newValue               the value which should be saved into LDAP (i.e. NOT processed by
     *                               {@link DirectoryAttributeRetriever#toSaveableLDAPValue(String)} yet)
     */
    protected static ModificationItem createModificationItem(String directoryAttributeName, String oldValue, String newValue) {
        // do some manual dirty checking
        if (oldValue == null && newValue == null) {
            // no modification
            return null;
        } else if (oldValue == null) { // need to create the item
            return new ModificationItem(DirContext.ADD_ATTRIBUTE, new BasicAttribute(directoryAttributeName,
                    DirectoryAttributeRetriever.toSaveableLDAPValue(newValue)));
        /*
        // NOTE: WE CURRENTLY HAVE NO NEED TO REMOVE ITEMS BECAUSE ALL THAT EXIST, EXIST PERMANENTLY OR GET REPLACED BY A SPACE
        else if (newValue == null)
        {
            // need to remove the item
            return new ModificationItem(DirContext.REMOVE_ATTRIBUTE, new BasicAttribute(attributeName));
        }
        */
        } else if (!oldValue.equals(newValue)) {
            // need to update the item
            return new ModificationItem(DirContext.REPLACE_ATTRIBUTE, new BasicAttribute(directoryAttributeName,
                    DirectoryAttributeRetriever.toSaveableLDAPValue(newValue)));
        } else {
            // attribute hasn't changed
            return null;
        }
    }

    /////////////////// CONTEXT MAPPERS ///////////////////

    /**
     * Returns a ContextMapper that can transform a Context into a User.
     *
     * @return a ContextMapper that can transform a Context into a User.
     */
    public ContextMapperWithRequiredAttributes<LDAPUserWithAttributes> getUserContextMapper() {
        return new UserContextMapper(this.getDirectoryId(), ldapPropertiesMapper, getCustomUserAttributeMappers());
    }

    /**
     * @return a collection of custom attribute mappers. By default just return an empty list.
     */
    protected List<AttributeMapper> getCustomUserAttributeMappers() {
        return Collections.emptyList();
    }

    /**
     * Returns a ContextMapper ready to translate LDAP objects into Groups and fetches all member objects.
     *
     * @param groupType the GroupType
     * @return a ContextMapper ready to translate LDAP objects into Groups and fetches all member objects
     */
    public ContextMapperWithRequiredAttributes<LDAPGroupWithAttributes> getGroupContextMapper(GroupType groupType) {
        Validate.notNull(groupType, "group type cannot be null");
        return new GroupContextMapper(getDirectoryId(), groupType, ldapPropertiesMapper, getCustomGroupAttributeMappers());
    }

    /**
     * As a minimum, this SHOULD provide an attribute mapper that maps the group members attribute (if available).
     *
     * @return collection of custom attribute mappers (cannot be <tt>null</tt> but can be an empty list).
     */
    protected List<AttributeMapper> getCustomGroupAttributeMappers() {
        return Collections.emptyList();
    }

    /////////////////// USER OPERATIONS ///////////////////

    @Override
    public LDAPUserWithAttributes findUserByName(String name) throws UserNotFoundException, OperationFailedException {
        Validate.notNull(name, "name argument cannot be null");
        // equivalent call to findUserWithAttributes
        return findUserWithAttributesByName(name);

        // TODO: potential performance benefit: request only the first-class attributes
    }

    @Override
    public LDAPUserWithAttributes findUserWithAttributesByName(final String name) throws UserNotFoundException, OperationFailedException {
        Validate.notNull(name, "name argument cannot be null");

        EntityQuery<User> query = QueryBuilder.queryFor(User.class, EntityDescriptor.user()).with(Restriction.on(UserTermKeys.USERNAME).exactlyMatching(name)).returningAtMost(1);

        final List<LDAPUserWithAttributes> users;

        users = searchUserObjects(query);

        if (users.isEmpty()) {
            throw new UserNotFoundException(name);
        }

        // return the first object
        return users.get(0);
    }

    @Override
    public LDAPUserWithAttributes findUserByExternalId(final String externalId) throws UserNotFoundException, OperationFailedException {
        Validate.notNull(externalId, "externalId argument cannot be null");

        EntityQuery<User> query = QueryBuilder.queryFor(User.class, EntityDescriptor.user()).with(Restriction.on(UserTermKeys.EXTERNAL_ID).exactlyMatching(externalId)).returningAtMost(1);

        final List<LDAPUserWithAttributes> users = searchUserObjects(query);

        if (users.isEmpty()) {
            UserNotFoundException.throwNotFoundByExternalId(externalId);
        }

        // return the first object
        return users.get(0);
    }

    protected List<LDAPUserWithAttributes> searchUserObjects(EntityQuery<?> query) throws OperationFailedException, IllegalArgumentException {
        if (query == null) {
            throw new IllegalArgumentException("user search can only evaluate non-null EntityQueries for Entity.USER");
        }

        if (query.getEntityDescriptor().getEntityType() != Entity.USER) {
            throw new IllegalArgumentException("user search can only evaluate EntityQueries for Entity.USER");
        }

        Name baseDN = searchDN.getUser();

        List<LDAPUserWithAttributes> results;

        try {
            LDAPQuery ldapQuery = ldapQueryTranslater.asLDAPFilter(query, ldapPropertiesMapper);
            String filter = ldapQuery.encode();
            logger.debug("Performing user search: baseDN = " + baseDN + " - filter = " + filter);

            results = searchEntities(baseDN, filter, getUserContextMapper(), query.getStartIndex(), query.getMaxResults());

        } catch (NullResultException e) {
            results = Collections.emptyList();
        }

        return results;
    }

    @Override
    public void removeUser(String name) throws UserNotFoundException, OperationFailedException {
        Validate.notEmpty(name, "name argument cannot be null or empty");

        LDAPUserWithAttributes user = findUserByName(name);

        // remove the user by dn
        try {
            ldapTemplate.unbind(asLdapUserName(user.getDn(), name));
        } catch (NamingException ex) {
            throw new OperationFailedException(ex);
        }
    }

    @Override
    public void updateUserCredential(String name, PasswordCredential credential) throws InvalidCredentialException, UserNotFoundException, OperationFailedException {
        Validate.notEmpty(name, "name argument cannot be null or empty");
        Validate.notNull(credential, "credential argument cannot be null");

        if (credential.getCredential() == null) {
            throw new InvalidCredentialException("Credential's value must not be null");
        }

        ModificationItem[] mods = new ModificationItem[1];

        Object encodedCredential = getCredentialEncoder().encodeCredential(credential);

        mods[0] = new ModificationItem(DirContext.REPLACE_ATTRIBUTE,
                new BasicAttribute(ldapPropertiesMapper.getUserPasswordAttribute(), encodedCredential));

        LdapName userDn = asLdapUserName(findUserByName(name).getDn(), name);

        try {
            ldapTemplate.modifyAttributes(userDn, mods);
        } catch (NamingException ex) {
            throw new OperationFailedException(ex);
        }
    }

    @Override
    public User renameUser(final String oldName, final String newName) throws UserNotFoundException, InvalidUserException, OperationFailedException {
        // TODO: support this, aka CWD-1466
        throw new OperationNotSupportedException("User renaming is not supported for LDAP directories.");
    }

    @Override
    public void storeUserAttributes(final String username, final Map<String, Set<String>> attributes) throws UserNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException("Custom user attributes are not yet supported for LDAP directories");
    }

    @Override
    public void removeUserAttributes(final String username, final String attributeName) throws UserNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException("Custom user attributes are not yet supported for LDAP directories");
    }

    /**
     * Translates the <code>User</code> into LDAP attributes, in preparation for creating a new user.
     *
     * @param user       The user object to translate into LDAP attributes
     * @param credential raw password.
     * @return An Attributes object populated with directory-specific information.
     * @throws InvalidCredentialException The password, if supplied, was invalid in some manner.
     * @throws NamingException            If the <code>User</code> could not be translated to an <code>Attributes</code>
     */
    protected Attributes getNewUserAttributes(User user, PasswordCredential credential) throws InvalidCredentialException, NamingException {
        // get the basic attributes
        LDAPUserAttributesMapper mapper = new LDAPUserAttributesMapper(getDirectoryId(), ldapPropertiesMapper);
        Attributes attributes = mapper.mapAttributesFromUser(user);

        if (credential != null && credential.getCredential() != null) {
            Object encodedCredential = getCredentialEncoder().encodeCredential(credential);
            attributes.put(ldapPropertiesMapper.getUserPasswordAttribute(), encodedCredential);
        }

        // add directory-specific attributes to the user
        getNewUserDirectorySpecificAttributes(user, attributes);

        return attributes;
    }

    /**
     * Populates attributes object with directory-specific attributes.
     * <p>
     * Overrider of this method can take advantage of the default group attributes mapping logic
     * in {#getNewUserAttributes(User)}.
     * <p>
     * Note that the attribute values supplied here will be used raw. This entails that overrider is responsible
     * for supplying values in a format supported by the directory.
     * In some directory implementations, for example, a blank string ("") is considered illegal. Overrider thus
     * would have to make sure the method does not generate a value as such.
     *
     * @param user       (potential) source of information that needs to be added.
     * @param attributes attributes to add directory-specific information to.
     */
    protected void getNewUserDirectorySpecificAttributes(User user, Attributes attributes) {
        // default is a no-op
    }

    /**
     * Adds a user to LDAP.
     *
     * @param user       template of the user to add.
     * @param credential password.
     * @return LDAP user retrieved from LDAP after successfully adding the user to LDAP.
     * @throws InvalidUserException       if the user to create was deemed invalid by the LDAP server or already exists.
     * @throws InvalidCredentialException if the password credential was deemed invalid by the password encoder.
     * @throws OperationFailedException   if we were unable to add the user to LDAP.
     */
    @Override
    public LDAPUserWithAttributes addUser(UserTemplate user, PasswordCredential credential)
            throws InvalidUserException, InvalidCredentialException, OperationFailedException {
        return addUser(UserTemplateWithAttributes.toUserWithNoAttributes(user), credential);
    }

    /**
     * Adds a user to LDAP.
     *
     * @param user       template of the user to add.
     * @param credential password.
     * @return LDAP user retrieved from LDAP after successfully adding the user to LDAP.
     * @throws InvalidUserException       if the user to create was deemed invalid by the LDAP server or already exists.
     * @throws InvalidCredentialException if the password credential was deemed invalid by the password encoder.
     * @throws OperationFailedException   if we were unable to add the user to LDAP.
     */
    @Override
    public LDAPUserWithAttributes addUser(UserTemplateWithAttributes user, PasswordCredential credential)
            throws InvalidUserException, InvalidCredentialException, OperationFailedException {
        Validate.notNull(user, "user cannot be null");
        Validate.notNull(user.getName(), "user.name cannot be null");
        try {
            // build the DN for the new user
            LdapName dn = nameConverter.getName(ldapPropertiesMapper.getUserNameRdnAttribute(), user.getName(), searchDN.getUser());

            // This call currently discards the attributes potentially attached to the current user, since these are not persisted at present
            Attributes attrs = getNewUserAttributes(user, credential);

            // create the user
            ldapTemplate.bind(dn, null, attrs);

            return findEntityByDN(getStandardisedDN(dn), LDAPUserWithAttributes.class);
        } catch (NamingException e) {
            throw new InvalidUserException(user, e.getMessage(), e);
        } catch (InvalidNameException e) {
            throw new InvalidUserException(user, e.getMessage(), e);
        } catch (GroupNotFoundException e) {
            throw new AssertionError("Should not throw a GroupNotFoundException");
        } catch (UserNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    /**
     * A default install of many directory servers (inc. Sun DSEE 6.2 and Apache DS 1.0.2) requires the following to be
     * set before user creation is allowed:
     * <pre>
     * objectClass -&gt; inetorgperson
     * cn          -&gt;
     * sn          -&gt;
     * </pre>
     * If a call is being made from an external system (eg JIRA), the user is created with the bare minimum of
     * attributes, then later updated. We need to make sure to add <code>sn</code> if it's not present in the
     * information provided.
     *
     * @param attrs          The LDAP user attributes to be checked and potentially updated.
     * @param defaultSnValue default lastname/surname value
     */
    protected void addDefaultSnToUserAttributes(Attributes attrs, String defaultSnValue) {
        addDefaultValueToUserAttributesForAttribute(ldapPropertiesMapper.getUserLastNameAttribute(), attrs, defaultSnValue);
    }

    protected void addDefaultValueToUserAttributesForAttribute(String attributeName, Attributes attrs, String defaultValue) {
        if (attrs == null) {
            return;
        }

        Attribute userAttribute = attrs.get(attributeName);
        if (userAttribute == null) {
            attrs.put(new BasicAttribute(attributeName, defaultValue));
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T extends LDAPDirectoryEntity> T findEntityByDN(String dn, Class<T> entityClass)
            throws UserNotFoundException, GroupNotFoundException, OperationFailedException {
        dn = standardiseDN(dn);
        if (User.class.isAssignableFrom(entityClass)) {
            return findEntityByDN(dn, getStandardisedDN(searchDN.getUser()), ldapPropertiesMapper.getUserFilter(), getUserContextMapper(), entityClass);
        } else if (Group.class.isAssignableFrom(entityClass)) {
            LDAPDirectoryEntity groupEntity = findEntityByDN(dn, getStandardisedDN(searchDN.getGroup()), ldapPropertiesMapper.getGroupFilter(), getGroupContextMapper(GroupType.GROUP), entityClass);
            return (T) postprocessGroups(Collections.singletonList((LDAPGroupWithAttributes) groupEntity)).get(0);
        } else {
            throw new IllegalArgumentException("Class " + entityClass.getCanonicalName() + " is not assignable from " + User.class.getCanonicalName() + " or " + Group.class.getCanonicalName());
        }
    }

    protected <T extends LDAPDirectoryEntity> RuntimeException typedEntityNotFoundException(String name, Class<T> entityClass)
            throws UserNotFoundException, GroupNotFoundException {
        if (User.class.isAssignableFrom(entityClass)) {
            throw new UserNotFoundException(name);
        } else if (Group.class.isAssignableFrom(entityClass)) {
            throw new GroupNotFoundException(name);
        } else {
            throw new IllegalArgumentException("Class " + entityClass.getCanonicalName() + " is not assignable from " + User.class.getCanonicalName() + " or " + Group.class.getCanonicalName());
        }
    }

    @SuppressWarnings("unchecked")
    protected <T extends LDAPDirectoryEntity> T findEntityByDN(String dn, String baseDN, String filter, ContextMapper contextMapper, Class<T> entityClass)
            throws UserNotFoundException, GroupNotFoundException, OperationFailedException {
        if (StringUtils.isBlank(dn)) {
            throw typedEntityNotFoundException("Blank DN", entityClass);
        }

        if (dn.endsWith(baseDN)) {
            if (logger.isDebugEnabled()) {
                logger.debug("Executing search at DN: <" + dn + "> with filter: <" + filter + ">");
            }

            List<T> entities = null;

            try {
                SearchControls searchControls = new SearchControls();
                searchControls.setSearchScope(SearchControls.OBJECT_SCOPE);
                searchControls.setTimeLimit(ldapPropertiesMapper.getSearchTimeLimit());
                searchControls.setReturningObjFlag(false); // should never ever be set to true, see CWD-4754 for more details
                entities = ldapTemplate.search(asLdapName(dn, "DN: " + dn, entityClass), filter, searchControls, contextMapper);
            } catch (NameNotFoundException e) {
                // SpringLDAP likes to chuck this exception, essentially an ONFE

                if (logger.isDebugEnabled()) {
                    logger.debug("Search failed", e);
                }

                // entities == null so we'll get a nice ONFE just below
            } catch (NamingException ex) {
                throw new OperationFailedException(ex);
            }

            if (entities != null && !entities.isEmpty()) {
                return entities.get(0);
            } else {
                if (logger.isDebugEnabled()) {
                    logger.debug("Entity DN <" + dn + "> does not exist or does not match filter <" + filter + ">");
                }
                throw typedEntityNotFoundException("DN: " + dn, entityClass);
            }
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug("Entity DN <" + dn + "> is outside the entity base DN subtree scope <" + baseDN + ">");
            }
            throw typedEntityNotFoundException("DN: " + dn, entityClass);
        }
    }

    @Override
    public User updateUser(UserTemplate user) throws UserNotFoundException, OperationFailedException {
        Validate.notNull(user, "user cannot be null");
        Validate.isTrue(StringUtils.isNotBlank(user.getName()), "user cannot have blank user name");

        // pre-populate user names (first name, last name, display name may need to be constructed)
        User populatedUser = UserUtils.populateNames(user);

        // Get the current user, if this user is not found a ONFE will be thrown
        LDAPUserWithAttributes currentUser = findUserByName(user.getName());

        // get ldap helper object type
        String ldapUserObjectType = ldapPropertiesMapper.getUserObjectClass();

        // TODO: This should probably be removed, but it is really redundant
        // if ldap object type is of inetOrgPerson or AD User object, we can update 'certain' attributes
        if ("inetOrgPerson".equalsIgnoreCase(ldapUserObjectType) || "user".equalsIgnoreCase(ldapUserObjectType)) {
            List<ModificationItem> modificationItems = getUserModificationItems(populatedUser, currentUser);

            // Perform the update if there are modification items
            if (!modificationItems.isEmpty()) {
                try {
                    ldapTemplate.modifyAttributes(asLdapUserName(currentUser.getDn(), user.getName()), modificationItems.toArray(new ModificationItem[modificationItems.size()]));
                } catch (NamingException ex) {
                    throw new OperationFailedException(ex);
                }
            }
        }

        // Return the user 'fresh' from the LDAP directory
        try {
            return findEntityByDN(currentUser.getDn(), LDAPUserWithAttributes.class);
        } catch (GroupNotFoundException e) {
            throw new AssertionError("Should not throw a GroupNotFoundException");
        }
    }

    protected List<ModificationItem> getUserModificationItems(User userTemplate, LDAPUserWithAttributes currentUser) {
        List<ModificationItem> modificationItems = new ArrayList<ModificationItem>();

        // sn
        ModificationItem snMod = createModificationItem(ldapPropertiesMapper.getUserLastNameAttribute(),
                currentUser.getLastName(), userTemplate.getLastName());
        if (snMod != null) {
            modificationItems.add(snMod);
        }

        // mail
        ModificationItem mailMod = createModificationItem(ldapPropertiesMapper.getUserEmailAttribute(),
                currentUser.getEmailAddress(), userTemplate.getEmailAddress());
        if (mailMod != null) {
            modificationItems.add(mailMod);
        }

        // giveName
        ModificationItem givenNameMod = createModificationItem(ldapPropertiesMapper.getUserFirstNameAttribute(),
                currentUser.getFirstName(), userTemplate.getFirstName());
        if (givenNameMod != null) {
            modificationItems.add(givenNameMod);
        }

        // displayName
        ModificationItem displayNameMod = createModificationItem(ldapPropertiesMapper.getUserDisplayNameAttribute(),
                currentUser.getDisplayName(), userTemplate.getDisplayName());
        if (displayNameMod != null) {
            modificationItems.add(displayNameMod);
        }

        return modificationItems;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> List<T> searchUsers(EntityQuery<T> query) throws OperationFailedException {
        List<LDAPUserWithAttributes> users = searchUserObjects(query);

        if (query.getReturnType() == String.class) { // as names
            return (List<T>) convertEntitiesToNames(users);
        } else {
            return (List<T>) users;
        }
    }

    @Override
    public User authenticate(String name, PasswordCredential credential)
            throws InvalidAuthenticationException, UserNotFoundException, OperationFailedException {
        // connection object
        LdapContextSource ctxSource = new LdapContextSource();

        // connection url
        ctxSource.setUrl(ldapPropertiesMapper.getConnectionURL());

        // username/password
        LDAPUserWithAttributes user = findUserByName(name);
        ctxSource.setUserDn(user.getDn());

        logger.debug("Authenticating user '{}' with DN '{}'", name, user.getDn());

        // Do not allow the password credential to be blank
        // We should possibly follow http://opensource.atlassian.com/projects/spring/browse/LDAP-39
        // so we can simplify this call down. Currently if you use a blank password the below call
        // to getReadWriteContext will succeed: http://jira.atlassian.com/browse/CWD-316
        if (credential == null || StringUtils.isBlank(credential.getCredential())) {
            throw new InvalidAuthenticationException("You cannot authenticate with a blank password");
        }

        if (credential.isEncryptedCredential()) {
            throw new InvalidAuthenticationException("You cannot authenticate with an encrypted PasswordCredential");
        }

        ctxSource.setPassword(credential.getCredential());

        // additional ldap properties
        ctxSource.setBaseEnvironmentProperties(getBaseEnvironmentProperties());

        ctxSource.setPooled(false);

        try {
            ctxSource.afterPropertiesSet();

            // Perform the authentication call by getting the context with the above attributes
            DirContext ctxt = ctxSource.getReadWriteContext();
            ctxt.close();
        } catch (NamingException e) {
            throw InvalidAuthenticationException.newInstanceWithNameAndDescriptionFromCause(name, e);
        } catch (Exception e) {
            throw new InvalidAuthenticationException(name, e);
        }

        return user;
    }


    /////////////////// GROUP OPERATIONS ///////////////////

    @Override
    public LDAPGroupWithAttributes findGroupByName(String name) throws GroupNotFoundException, OperationFailedException {
        Validate.notNull(name, "name argument cannot be null");
        // equivalent call to findGroupWithAttributes
        return findGroupWithAttributesByName(name);

        // TODO: potential performance benefit: request only the first-class attributes
    }

    @Override
    public LDAPGroupWithAttributes findGroupWithAttributesByName(final String name) throws GroupNotFoundException, OperationFailedException {
        Validate.notNull(name, "name argument cannot be null");

        EntityQuery<Group> query = QueryBuilder.queryFor(Group.class, EntityDescriptor.group()).with(Restriction.on(GroupTermKeys.NAME).exactlyMatching(name)).returningAtMost(1);

        try {
            // query is constrained to returnAtMost(1)
            return Iterables.getOnlyElement(searchGroupObjects(query, getGroupContextMapper(GroupType.GROUP)));
        } catch (NoSuchElementException e) {
            throw new GroupNotFoundException(name);
        }
    }

    protected LDAPGroupWithAttributes findGroupByNameAndType(final String name, GroupType groupType) throws GroupNotFoundException, OperationFailedException {
        Validate.notNull(name, "name argument cannot be null");

        EntityQuery<Group> query = QueryBuilder.queryFor(Group.class, EntityDescriptor.group(groupType)).with(Restriction.on(GroupTermKeys.NAME).exactlyMatching(name)).returningAtMost(1);

        ContextMapperWithRequiredAttributes<LDAPGroupWithAttributes> mapper;
        if (groupType == null) {
            mapper = getGroupContextMapper(GroupType.GROUP);
        } else {
            mapper = getGroupContextMapper(groupType);
        }

        try {
            // query is constrained to returnAtMost(1)
            return Iterables.getOnlyElement(searchGroupObjects(query, mapper));
        } catch (NoSuchElementException e) {
            throw new GroupNotFoundException(name);
        }
    }

    /**
     * This method expects that the query contains a non-null groupType in the entityDescriptor.
     *
     * @param query search query.
     * @return list of results.
     * @throws OperationFailedException represents a Communication error when trying to talk to a remote directory
     */
    protected <T> List<T> searchGroupObjectsOfSpecifiedGroupType(EntityQuery<?> query, ContextMapperWithRequiredAttributes<T> mapper) throws OperationFailedException {
        GroupType groupType = query.getEntityDescriptor().getGroupType();

        Name baseDN;
        if (GroupType.GROUP.equals(groupType)) {
            baseDN = searchDN.getGroup();
        } else if (GroupType.LEGACY_ROLE.equals(groupType)) {
            return Collections.emptyList();
        } else {
            throw new IllegalArgumentException("Cannot search for groups of type: " + groupType);
        }

        List<T> results;

        try {
            LDAPQuery ldapQuery = ldapQueryTranslater.asLDAPFilter(query, ldapPropertiesMapper);
            String filter = ldapQuery.encode();
            logger.debug("Performing group search: baseDN = " + baseDN + " - filter = " + filter);

            results = searchEntities(baseDN, filter, mapper, query.getStartIndex(), query.getMaxResults());
        } catch (NullResultException e) {
            results = Collections.emptyList();
        }

        return results;
    }

    protected <T> Iterable<T> searchGroupObjects(EntityQuery<?> query, ContextMapperWithRequiredAttributes<T> mapper)
            throws OperationFailedException {
        Validate.notNull(query, "query argument cannot be null");

        if (query.getEntityDescriptor().getEntityType() != Entity.GROUP) {
            throw new IllegalArgumentException("group search can only evaluate EntityQueries for Entity.GROUP");
        }

        GroupType groupType = query.getEntityDescriptor().getGroupType();

        if (groupType == null) {
            int groupStartIndex = query.getStartIndex();
            int groupMaxResults = query.getMaxResults();

            // search for groups
            GroupQuery<Group> groupQuery = new GroupQuery<Group>(Group.class, GroupType.GROUP, query.getSearchRestriction(), groupStartIndex, groupMaxResults);
            List<T> results = ImmutableList.copyOf(searchGroupObjectsOfSpecifiedGroupType(groupQuery, mapper));

            return constrainResults(results, query.getStartIndex(), query.getMaxResults());
        } else {
            // group or role was specified in the query, so we can pass the query straight through
            return searchGroupObjectsOfSpecifiedGroupType(query, mapper);
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> List<T> searchGroups(EntityQuery<T> query) throws OperationFailedException {
        Validate.notNull(query, "query argument cannot be null");

        GroupType groupType = query.getEntityDescriptor().getGroupType();

        if (groupType == GroupType.LEGACY_ROLE) {
            return Collections.emptyList();
        } else if (groupType != null && groupType != GroupType.GROUP) {
            throw new IllegalArgumentException("group search can only evaluate EntityQueries for GroupType.GROUP");
        }

        if (query.getReturnType() == String.class) { // as names
            ContextMapperWithRequiredAttributes<NamedLdapEntity> mapper =
                    NamedLdapEntity.mapperFromAttribute(ldapPropertiesMapper.getGroupNameAttribute());
            Iterable<String> namedLdapEntities = NamedLdapEntity.namesOf(searchGroupObjects(query, mapper));
            return (List<T>) ImmutableList.copyOf(namedLdapEntities);
        } else {
            ContextMapperWithRequiredAttributes<LDAPGroupWithAttributes> mapper =
                    getGroupContextMapper(GroupType.GROUP);
            return (List<T>) ImmutableList.copyOf(searchGroupObjects(query, mapper));
        }
    }

    /**
     * Perform any post-processing on groups.
     *
     * @param groups to post-process
     * @return list of groups that have been processed if required
     * @throws OperationFailedException if processing encounters a problem with the underlying directory
     */
    protected List<LDAPGroupWithAttributes> postprocessGroups(List<LDAPGroupWithAttributes> groups)
            throws OperationFailedException {
        return groups;
    }


    protected Attributes getNewGroupAttributes(Group group) throws NamingException {
        // get the basic attributes
        LDAPGroupAttributesMapper mapper = new LDAPGroupAttributesMapper(getDirectoryId(), group.getType(), ldapPropertiesMapper);
        Attributes attributes = mapper.mapAttributesFromGroup(group);

        // extension point: add directory-specific attributes to the group
        getNewGroupDirectorySpecificAttributes(group, attributes);

        // add member if required
        String defaultContainerMemberDN = getInitialGroupMemberDN();
        if (defaultContainerMemberDN != null) {
            // OpenLDAP fails if defaultContainerMemberDN is a space character, so don't pad it
            attributes.put(new BasicAttribute(ldapPropertiesMapper.getGroupMemberAttribute(), defaultContainerMemberDN));
        }

        return attributes;
    }

    /**
     * Populates attributes object with directory-specific attributes.
     * <p>
     * Overrider of this method can take advantage of the default group attributes mapping logic
     * in {#getNewGroupAttributes(Group)}.
     * <p>
     * Note that the attribute values supplied here will be used raw. This entails that overrider is responsible
     * for supplying values in a format supported by the directory.
     * In some directory implementations, for example, a blank string ("") is considered illegal. Overrider thus
     * would have to make sure the method does not generate a value as such.
     *
     * @param group      (potential) source of information that needs to be added.
     * @param attributes attributes to add directory-specific information to.
     */
    protected void getNewGroupDirectorySpecificAttributes(final Group group, final Attributes attributes) {
        // default no-op
    }

    /**
     * Returns the default container member DN.
     * <p>
     * If this method returns null or blank string, no member DN will be added.
     *
     * @return empty member.
     */
    protected String getInitialGroupMemberDN() {
        // empty member
        return "";
    }

    @Override
    public Group addGroup(GroupTemplate group) throws InvalidGroupException, OperationFailedException {
        Validate.notNull(group, "group cannot be null");
        Validate.isTrue(StringUtils.isNotBlank(group.getName()), "group cannot have blank group name");

        if (groupExists(group)) {
            throw new InvalidGroupException(group, "Group already exists");
        }

        LdapName baseDN;
        String nameAttribute;
        if (group.getType() == GroupType.GROUP) {
            baseDN = searchDN.getGroup();
            nameAttribute = ldapPropertiesMapper.getGroupNameAttribute();
        } else {
            throw new InvalidGroupException(group, "group.type must be GroupType.GROUP");
        }

        try {
            LdapName dn = nameConverter.getName(nameAttribute, group.getName(), baseDN);

            Attributes groupAttributes = getNewGroupAttributes(group);

            // create the group
            ldapTemplate.bind(dn, null, groupAttributes);

            return findEntityByDN(getStandardisedDN(dn), LDAPGroupWithAttributes.class);
        } catch (UserNotFoundException e) {
            throw new AssertionError("Should not throw UserNotFoundException");
        } catch (GroupNotFoundException e) {
            throw new OperationFailedException(e);
        } catch (NamingException e) {
            throw new InvalidGroupException(group, e.getMessage(), e);
        } catch (InvalidNameException e) {
            throw new InvalidGroupException(group, e.getMessage(), e);
        }
    }

    @Override
    public Group updateGroup(GroupTemplate group) throws GroupNotFoundException, OperationFailedException {
        Validate.notNull(group, "group cannot be null");
        Validate.isTrue(StringUtils.isNotBlank(group.getName()), "group cannot have blank group name");

        // Get the current group, if this group is not found a ONFE will be thrown
        LDAPGroupWithAttributes currentGroup = findGroupByName(group.getName());

        if (currentGroup.getType() != group.getType()) {
            throw new OperationNotSupportedException("Cannot modify the GroupType for an LDAP group");
        }

        List<ModificationItem> modificationItems = new ArrayList<ModificationItem>();

        // description
        String descriptionAttribute;
        if (group.getType() == GroupType.GROUP) {
            descriptionAttribute = ldapPropertiesMapper.getGroupDescriptionAttribute();
        } else {
            descriptionAttribute = ldapPropertiesMapper.getRoleDescriptionAttribute();
        }

        ModificationItem descriptionMod = createModificationItem(descriptionAttribute, currentGroup.getDescription(),
                group.getDescription());
        if (descriptionMod != null) {
            modificationItems.add(descriptionMod);
        }

        // Perform the update if there are modification items
        if (!modificationItems.isEmpty()) {
            try {
                ldapTemplate.modifyAttributes(asLdapGroupName(currentGroup.getDn(), group.getName()), modificationItems.toArray(new ModificationItem[modificationItems.size()]));
            } catch (NamingException ex) {
                throw new OperationFailedException(ex);
            }
        }

        // Return the group 'fresh' from the LDAP directory
        try {
            return findEntityByDN(currentGroup.getDn(), LDAPGroupWithAttributes.class);
        } catch (UserNotFoundException e) {
            throw new AssertionError("Should not throw UserNotFoundException.");
        }
    }

    @Override
    public void removeGroup(String name) throws GroupNotFoundException, OperationFailedException {
        Validate.notEmpty(name, "name argument cannot be null or empty");

        LDAPGroupWithAttributes group = findGroupByName(name);

        // remove the group by dn
        try {
            ldapTemplate.unbind(asLdapGroupName(group.getDn(), name));
        } catch (NamingException ex) {
            throw new OperationFailedException(ex);
        }
    }

    @Override
    public Group renameGroup(final String oldName, final String newName) throws GroupNotFoundException, InvalidGroupException, OperationFailedException {
        // TODO: support this, aka CWD-1466
        throw new OperationNotSupportedException("Group renaming is not yet supported for LDAP directories");
    }

    @Override
    public void storeGroupAttributes(final String groupName, final Map<String, Set<String>> attributes) throws GroupNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException("Custom group attributes are not yet supported for LDAP directories");
    }

    @Override
    public void removeGroupAttributes(final String groupName, final String attributeName) throws GroupNotFoundException, OperationFailedException {
        throw new OperationNotSupportedException("Custom group attributes are not yet supported for LDAP directories");
    }

    @Override
    public <T> List<T> searchGroupRelationships(final MembershipQuery<T> query) throws OperationFailedException {
        Validate.notNull(query, "query argument cannot be null");

        if (query.getEntityToMatch().getEntityType() == Entity.GROUP && query.getEntityToReturn().getEntityType() == Entity.GROUP && query.getEntityToMatch().getEntityType() != query.getEntityToReturn().getEntityType()) {
            throw new IllegalArgumentException("Cannot search for group relationships of mismatching GroupTypes: attempted to match <" + query.getEntityToMatch().getEntityType() + "> and return <" + query.getEntityToReturn().getEntityType() + ">");
        }

        if (query.getSearchRestriction() != NullRestrictionImpl.INSTANCE) {
            throw new IllegalArgumentException("LDAP-based membership queries do not support search restrictions.");
        }

        Iterable<T> results;

        if (query.getEntityToMatch().getEntityType() == Entity.GROUP && query.getEntityToReturn().getEntityType() == Entity.USER) {
            GroupType groupType = query.getEntityToMatch().getGroupType();

            if (groupType == null) {
                // if groupType is null then we are searching for either the group or the role (try group first, then role)
                MembershipQuery<T> groupQuery = QueryBuilder.createMembershipQuery(query.getMaxResults(), query.getStartIndex(), query.isFindChildren(), query.getEntityToReturn(), query.getReturnType(), query.getEntityToMatch(), query.getEntityNameToMatch());
                results = searchGroupRelationshipsWithGroupTypeSpecified(groupQuery);
            } else {
                // groupType has been specified, so safe to execute directly
                results = searchGroupRelationshipsWithGroupTypeSpecified(query);
            }
        } else if (query.getEntityToMatch().getEntityType() == Entity.USER && query.getEntityToReturn().getEntityType() == Entity.GROUP) {
            GroupType groupType = query.getEntityToReturn().getGroupType();

            if (groupType == null) {
                // if groupType is null then we are searching for either the group or the role (try group first, then role)
                MembershipQuery<T> groupQuery = QueryBuilder.createMembershipQuery(query.getMaxResults(), query.getStartIndex(), query.isFindChildren(), EntityDescriptor.group(GroupType.GROUP), query.getReturnType(), query.getEntityToMatch(), query.getEntityNameToMatch());
                results = searchGroupRelationshipsWithGroupTypeSpecified(groupQuery);
            } else {
                // groupType has been specified, so safe to execute directly
                results = searchGroupRelationshipsWithGroupTypeSpecified(query);
            }
        } else if (query.getEntityToMatch().getEntityType() == Entity.GROUP && query.getEntityToReturn().getEntityType() == Entity.GROUP) {
            GroupType groupTypeToMatch = query.getEntityToMatch().getGroupType();
            GroupType groupTypeToReturn = query.getEntityToReturn().getGroupType();

            if (groupTypeToMatch != groupTypeToReturn) {
                throw new IllegalArgumentException("Cannot search for group relationships of mismatching GroupTypes: attempted to match <" + groupTypeToMatch + "> and return <" + groupTypeToReturn + ">");
            }

            if (groupTypeToReturn == null) {
                // if groupType is null then we are searching for either the group or the role (try group first, then role)
                final MembershipQuery<T> groupQuery = QueryBuilder.createMembershipQuery(query.getMaxResults(), query.getStartIndex(), query.isFindChildren(), EntityDescriptor.group(GroupType.GROUP), query.getReturnType(), EntityDescriptor.group(GroupType.GROUP), query.getEntityNameToMatch());
                results = searchGroupRelationshipsWithGroupTypeSpecified(groupQuery);
            } else {
                // groupType has been specified, so safe to execute directly
                results = searchGroupRelationshipsWithGroupTypeSpecified(query);
            }
        } else {
            throw new IllegalArgumentException("Cannot search for relationships between a USER and another USER");
        }

        return ImmutableList.copyOf(results);
    }

    /**
     * Execute the search for group relationships given that a group of type GROUP or LEGACY_ROLE has
     * been specified in the EntityDescriptor for the group(s).
     *
     * @param query membership query with all GroupType's not null.
     * @return list of members or memberships depending on the query.
     * @throws OperationFailedException if the operation failed due to a communication error with the remote directory,
     *                                  or if the query is invalid
     */
    protected abstract <T> Iterable<T> searchGroupRelationshipsWithGroupTypeSpecified(MembershipQuery<T> query) throws OperationFailedException;

    /////////////////// MISCELLANEOUS OPERATIONS /////////////////

    /**
     * @return the credential encoder to use with this directory; must not be null.
     */
    protected abstract LDAPCredentialEncoder getCredentialEncoder();

    @Override
    public boolean supportsSettingEncryptedCredential() {
        return getCredentialEncoder().supportsSettingEncryptedPasswords();
    }

    /**
     * We don't support expiring passwords in LDAP directories (yet).
     *
     * @return {@code false}, always.
     */
    @Override
    public boolean supportsPasswordExpiration() {
        return false;
    }

    @Override
    public void expireAllPasswords() throws OperationFailedException {
        throw new OperationFailedException("Crowd does not support expiring passwords in LDAP directories");
    }

    @Override
    public boolean supportsNestedGroups() {
        return !ldapPropertiesMapper.isNestedGroupsDisabled();
    }

    @Override
    public boolean isRolesDisabled() {
        return true;
    }

    @Override
    public void testConnection() throws OperationFailedException {
        ClassLoader existingTCCL = Thread.currentThread().getContextClassLoader();
        try {
            //Any custom JNDI classes will come from our own classloader (webapp), so avoid Felix when loading JNDI classes
            Thread.currentThread().setContextClassLoader(SpringLDAPConnector.class.getClassLoader());
            LdapContext ldapContext = (LdapContext) contextSource.getReadOnlyContext();
            try {
                ldapContext.getConnectControls();
            } finally {
                ldapContext.close();
            }
        } catch (Exception e) {
            throw new OperationFailedException(e.getMessage());
        } finally {
            Thread.currentThread().setContextClassLoader(existingTCCL);
        }
    }

    @VisibleForTesting
    final String getStandardisedDN(LdapName dn) throws OperationFailedException {
        return DNStandardiser.standardise(dn, !ldapPropertiesMapper.isRelaxedDnStandardisation());
    }

    final String standardiseDN(String dn) {
        return DNStandardiser.standardise(dn, !ldapPropertiesMapper.isRelaxedDnStandardisation());
    }

    /**
     * This method is required to wrap DN's into LdapNames as spring-ldap
     * doesn't correctly handle operations with String dn arguments.
     * <p>
     * This mainly affects the escaping of slashes in DNs.
     * <p>
     * The resulting javax.naming.Name is not designed to be used for
     * caching or comparisons, rather, it is to be used for direct
     * calls into spring-ldap's ldapTemplate.
     *
     * @param dn          string version of DN.
     * @param entityName  used if NotFoundException needs to be thrown.
     * @param entityClass in case there is a problem converting the dn into an LdapName a NotFoundException of this type (group/user) will be thrown.
     *                    Must implement User or Group, otherwise an IllegalArgumentException will be thrown.
     * @return LdapName for use with spring-ldap.
     * @throws UserNotFoundException  unable to construct LdapName for User.
     * @throws GroupNotFoundException unable to construct LdapName for Group.
     */
    @SuppressFBWarnings(value = "LDAP_INJECTION", justification = "No user input")
    protected <T extends LDAPDirectoryEntity> LdapName asLdapName(String dn, String entityName, Class<T> entityClass) throws UserNotFoundException, GroupNotFoundException {
        try {
            return new LdapName(dn);
        } catch (InvalidNameException e) {
            throw typedEntityNotFoundException(entityName, entityClass);
        }
    }

    /**
     * Convenience method to convert group DN to LdapName,
     * throwing a GNFE with the supplied group name if unable
     * to construct the LdapName.
     *
     * @param dn        DN of the Group.
     * @param groupName for GNFE exception.
     * @return LdapName for DN.
     * @throws GroupNotFoundException unable to construct LdapName.
     */
    protected LdapName asLdapGroupName(String dn, String groupName) throws GroupNotFoundException {
        try {
            return asLdapName(dn, groupName, LDAPGroupWithAttributes.class);
        } catch (UserNotFoundException e) {
            throw new AssertionError("Should not throw UserNotFoundException.");
        }
    }

    /**
     * Convenience method to convert user DN to LdapName,
     * throwing a GNFE with the supplied user name if unable
     * to construct the LdapName.
     *
     * @param dn       DN of the User.
     * @param userName for GNFE exception.
     * @return LdapName for DN.
     * @throws UserNotFoundException unable to construct LdapName.
     */
    protected LdapName asLdapUserName(String dn, String userName) throws UserNotFoundException {
        try {
            return asLdapName(dn, userName, LDAPUserWithAttributes.class);
        } catch (GroupNotFoundException e) {
            throw new AssertionError("Should not throw GroupNotFoundException.");
        }
    }

    /**
     * Storing active/inactive flag for users in LDAP in general is currently not supported.
     *
     * @return false
     */
    @Override
    public boolean supportsInactiveAccounts() {
        return false;
    }

    @Override
    public RemoteDirectory getAuthoritativeDirectory() {
        return this;
    }

    /**
     * Returns <tt>true</tt> if the group exists.
     *
     * @param group Group to check
     * @return <tt>true</tt> if the group exists.
     * @throws OperationFailedException if the operation failed for any reason.
     */
    private boolean groupExists(final Group group) throws OperationFailedException {
        try {
            findGroupByName(group.getName());
            return true;
        } catch (GroupNotFoundException e) {
            return false;
        }
    }

    private static final DirContextProcessor DO_NOTHING_DIR_CONTEXT_PROCESSOR = new DirContextProcessor() {
        @Override
        public void preProcess(DirContext ctx) throws NamingException {

        }

        @Override
        public void postProcess(DirContext ctx) throws NamingException {
        }
    };

    protected ContextMapperWithRequiredAttributes<AvatarReference.BlobAvatar> avatarMapper() {
        return new JpegPhotoContextMapper();
    }

    @Override
    public AvatarReference.BlobAvatar getUserAvatarByName(String username, int sizeHint) throws OperationFailedException {
        EntityQuery<User> query = QueryBuilder.queryFor(User.class, EntityDescriptor.user()).with(Restriction.on(UserTermKeys.USERNAME).exactlyMatching(username)).returningAtMost(1);

        try {
            LDAPQuery ldapQuery = ldapQueryTranslater.asLDAPFilter(query, ldapPropertiesMapper);

            Name baseDN = searchDN.getUser();

            String filter = ldapQuery.encode();
            logger.debug("Performing user search: baseDN = " + baseDN + " - filter = " + filter);

            return Iterables.getFirst(
                    searchEntities(baseDN, filter, avatarMapper(), query.getStartIndex(), query.getMaxResults()),
                    null);
        } catch (NullResultException e) {
            return null;
        }
    }
}
