package com.atlassian.crowd.directory;

import java.util.Set;

import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;

import com.atlassian.crowd.directory.ldap.mapper.ContextMapperWithRequiredAttributes;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

import org.springframework.ldap.core.DirContextAdapter;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

class NamedLdapEntity {
    private final LdapName dn;
    private final String name;

    NamedLdapEntity(LdapName dn, String name) {
        this.dn = Preconditions.checkNotNull(dn, "DN may not be null");
        this.name = name;
    }

    LdapName getDn() {
        return dn;
    }

    String getName() {
        return name;
    }

    public String toString() {
        return dn.toString() + " = " + name;
    }


    static Iterable<String> namesOf(Iterable<? extends NamedLdapEntity> namedEntities) {
        return Iterables.transform(namedEntities, NamedLdapEntity::getName);
    }

    static Iterable<LdapName> dnsOf(Iterable<? extends NamedLdapEntity> namedEntities) {
        return Iterables.transform(namedEntities, NamedLdapEntity::getDn);
    }

    /**
     * A {@link ContextMapperWithRequiredAttributes} that captures only the name of an
     * entity from a single attribute.
     */
    static ContextMapperWithRequiredAttributes<NamedLdapEntity> mapperFromAttribute(String attrName) {
        return new NamedEntityMapper(attrName);
    }

    static class NamedEntityMapper implements ContextMapperWithRequiredAttributes<NamedLdapEntity> {
        private final String nameAttribute;

        NamedEntityMapper(String nameAttribute) {
            this.nameAttribute = Preconditions.checkNotNull(nameAttribute, "Attribute name may not be null");
        }

        @SuppressFBWarnings(value = "LDAP_INJECTION", justification = "No user input")
        @Override
        public NamedLdapEntity mapFromContext(Object ctx) {
            DirContextAdapter context = (DirContextAdapter) ctx;

            LdapName dn;

            try {
                dn = new LdapName(context.getDn().toString());
            } catch (InvalidNameException e) {
                throw new RuntimeException(e);
            }

            return new NamedLdapEntity(dn, context.getStringAttribute(nameAttribute));
        }

        @Override
        public Set<String> getRequiredLdapAttributes() {
            return ImmutableSet.of(nameAttribute);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            NamedEntityMapper that = (NamedEntityMapper) o;

            return nameAttribute.equals(that.nameAttribute);
        }

        @Override
        public int hashCode() {
            return nameAttribute.hashCode();
        }
    }
}
