package com.atlassian.crowd.console.value.directory;

import java.util.Map;

import com.atlassian.crowd.directory.SynchronisableDirectoryProperties;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.model.directory.DirectoryImpl;

import com.google.common.collect.ImmutableMap;

/**
 * Model for common settings for connector and crowd directories that can do synchronisation.
 */
public abstract class SynchronisableDirectoryConnection {
    private SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth syncGroupMembershipsAfterAuth = SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.DEFAULT;
    private boolean incrementalSyncEnabled = false;
    private long pollingIntervalInMin = 60;

    public void loadFromDirectory(Directory directory) {
        incrementalSyncEnabled = Boolean.parseBoolean(directory.getValue(SynchronisableDirectoryProperties.INCREMENTAL_SYNC_ENABLED));
        pollingIntervalInMin = getAttributeValueAsLong(directory, SynchronisableDirectoryProperties.CACHE_SYNCHRONISE_INTERVAL) / 60;
        syncGroupMembershipsAfterAuth = SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.forDirectory(directory);
    }

    public void updateDirectory(DirectoryImpl directory) {
        directory.setAttribute(SynchronisableDirectoryProperties.INCREMENTAL_SYNC_ENABLED, Boolean.toString(incrementalSyncEnabled));

        // convert pollingInterval to seconds to store in database
        directory.setAttribute(SynchronisableDirectoryProperties.CACHE_SYNCHRONISE_INTERVAL, Long.toString(pollingIntervalInMin * 60));

        directory.setAttribute(SynchronisableDirectoryProperties.SYNC_GROUP_MEMBERSHIP_AFTER_SUCCESSFUL_USER_AUTH_ENABLED, syncGroupMembershipsAfterAuth.getValue());
    }


    protected long getAttributeValueAsLong(Directory directory, String attributeName) {
        String value = directory.getValue(attributeName);
        return value == null ? 0 : Long.parseLong(value);
    }

    public boolean isIncrementalSyncEnabled() {
        return incrementalSyncEnabled;
    }

    public void setIncrementalSyncEnabled(boolean incrementalSyncEnabled) {
        this.incrementalSyncEnabled = incrementalSyncEnabled;
    }

    public long getPollingIntervalInMin() {
        return pollingIntervalInMin;
    }

    public void setPollingIntervalInMin(long pollingIntervalInMin) {
        this.pollingIntervalInMin = pollingIntervalInMin;
    }

    public String getSyncGroupMembershipsAfterAuth() {
        return syncGroupMembershipsAfterAuth.name();
    }

    public void setSyncGroupMembershipsAfterAuth(String syncGroupMembershipsAfterAuth) {
        this.syncGroupMembershipsAfterAuth = SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.valueOf(syncGroupMembershipsAfterAuth);
    }

    public Map<String, String> getSyncGroupMembershipsAfterAuthValues() {
        return ImmutableMap.of(
                SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.WHEN_AUTHENTICATION_CREATED_THE_USER.name(), "directoryconnector.update.groups.on.auth.whencreated.label",
                SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.NEVER.name(), "directoryconnector.update.groups.on.auth.never.label",
                SynchronisableDirectoryProperties.SyncGroupMembershipsAfterAuth.ALWAYS.name(), "directoryconnector.update.groups.on.auth.always.label"
        );
    }
}
