package com.atlassian.crowd.integration;

import java.security.Principal;
import java.util.Objects;

import com.google.common.base.Optional;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * This class holds authentication state.
 *
 * In addition to knowing whether a user is authenticated or not, this class
 * can optionally also contain the authenticated principal to save the caller
 * the trouble of finding the principal in another way. In other words, the
 * absence of a principal does not mean that the principal does not exist.
 *
 * @since 2.8.3
 */
public class AuthenticationState {
    private static final AuthenticationState AUTHENTICATED = new AuthenticationState(true, Optional.<Principal>absent());
    private static final AuthenticationState UNAUTHENTICATED = new AuthenticationState(false, Optional.<Principal>absent());

    private final boolean authenticated;
    private final Optional<Principal> authenticatedPrincipal;

    private AuthenticationState(boolean authenticated, Optional<Principal> authenticatedPrincipal) {
        this.authenticated = authenticated;
        this.authenticatedPrincipal = checkNotNull(authenticatedPrincipal);
    }

    /**
     * Create successful authentication state containing a user.
     *
     * @param principal authenticate user
     * @return successful authentication state containing a user
     */
    public static AuthenticationState authenticated(Principal principal) {
        return new AuthenticationState(true, Optional.of(principal));
    }

    /**
     * @return successful authentication state
     */
    public static AuthenticationState authenticated() {
        return AUTHENTICATED;
    }

    /**
     * @return unsuccessful authentication state
     */
    public static AuthenticationState unauthenticated() {
        return UNAUTHENTICATED;
    }

    /**
     * @return true if the user is authenticated
     */
    public boolean isAuthenticated() {
        return authenticated;
    }

    /**
     * @return the authenticated user if available
     */
    public Optional<Principal> getAuthenticatedPrincipal() {
        return authenticatedPrincipal;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AuthenticationState that = (AuthenticationState) o;
        return Objects.equals(authenticated, that.authenticated) &&
                Objects.equals(authenticatedPrincipal, that.authenticatedPrincipal);
    }

    @Override
    public int hashCode() {
        return Objects.hash(authenticated, authenticatedPrincipal);
    }
}
