package com.atlassian.crowd.model.user;

import com.atlassian.crowd.attribute.AttributeUtil;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Set;

public final class ImmutableUserWithAttributes extends BaseImmutableUser implements UserWithAttributes {
    private final ImmutableSetMultimap<String, String> attributes;

    private ImmutableUserWithAttributes(Builder builder) {
        super(builder);
        this.attributes = builder.attributes;
    }

    public static Builder builder(UserWithAttributes user) {
        return builder(user, AttributeUtil.toMultimap(user));
    }

    public static Builder builder(User user, SetMultimap<String, String> attributes) {
        return new ImmutableUserWithAttributes.Builder(user, attributes);
    }

    @Nonnull
    @Override
    public Set<String> getValues(String key) {
        return attributes.get(key);
    }

    @Nullable
    @Override
    public String getValue(String key) {
        return Iterables.getFirst(getValues(key), null);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    @Override
    public ImmutableUserWithAttributes withName(String name) {
        return builder(this).name(name).build();
    }

    public static class Builder extends BaseImmutableUser.Builder<Builder> {
        private ImmutableSetMultimap<String, String> attributes;

        public Builder(User user, SetMultimap<String, String> attributes) {
            super(user);
            setAttributes(attributes);
        }

        public Builder setAttributes(SetMultimap<String, String> attributes) {
            this.attributes = ImmutableSetMultimap.copyOf(attributes);
            return this;
        }

        @Override
        public ImmutableUserWithAttributes build() {
            return new ImmutableUserWithAttributes(this);
        }
    }
}
