package com.atlassian.crowd.model.user;

import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Date;
import java.util.Set;

public final class ImmutableTimestampedUserWithAttributes extends BaseImmutableUser implements UserWithAttributes, TimestampedUser {
    private final ImmutableSetMultimap<String, String> attributes;
    private final Date createdDate;
    private final Date updatedDate;

    private ImmutableTimestampedUserWithAttributes(Builder builder) {
        super(builder);
        this.attributes = builder.attributes;
        this.createdDate = builder.createdDate;
        this.updatedDate = builder.updatedDate;
    }

    public static Builder builder(TimestampedUser user, SetMultimap<String, String> attributes) {
        return new ImmutableTimestampedUserWithAttributes.Builder(user, attributes);
    }

    @Nonnull
    @Override
    public Set<String> getValues(String key) {
        return attributes.get(key);
    }

    @Nullable
    @Override
    public String getValue(String key) {
        return Iterables.getFirst(getValues(key), null);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }


    @Override
    public Date getCreatedDate() {
        return createdDate == null ? null : new Date(createdDate.getTime());
    }

    @Override
    public Date getUpdatedDate() {
        return updatedDate == null ? null : new Date(updatedDate.getTime());
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    @Override
    public ImmutableTimestampedUserWithAttributes withName(String name) {
        return builder(this, this.attributes).name(name).build();
    }

    public static class Builder extends BaseImmutableUser.Builder<Builder> {
        private ImmutableSetMultimap<String, String> attributes;
        private Date createdDate;
        private Date updatedDate;

        public Builder(TimestampedUser user, SetMultimap<String, String> attributes) {
            super(user);
            setAttributes(attributes);
            setCreatedDate(user.getCreatedDate());
            setUpdatedDate(user.getUpdatedDate());
        }

        public Builder setAttributes(SetMultimap<String, String> attributes) {
            this.attributes = ImmutableSetMultimap.copyOf(attributes);
            return this;
        }

        public Builder setCreatedDate(Date createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public Builder setUpdatedDate(Date updatedDate) {
            this.updatedDate = updatedDate;
            return this;
        }

        @Override
        public ImmutableTimestampedUserWithAttributes build() {
            return new ImmutableTimestampedUserWithAttributes(this);
        }
    }
}
