package com.atlassian.crowd.exception;

import com.atlassian.crowd.embedded.api.Directory;
import org.apache.commons.lang3.StringEscapeUtils;

import javax.annotation.Nullable;

/**
 * Thrown when the attempted authentication is not valid.
 */
public class InvalidAuthenticationException extends CrowdException {
    private String username;
    private Directory directory;

    /**
     * @since 4.4.0
     */
    public InvalidAuthenticationException(String username, Directory directory, Throwable cause) {
        super(String.format("Could not authenticate user %s in directory %s", username, directory.getName()), cause);
        this.username = username;
        this.directory = directory;
    }

    /**
     * Constructs a new <code>InvalidAuthenticationException</code> with the specified detail message.
     *
     * @param msg detail message
     */
    public InvalidAuthenticationException(final String msg) {
        super(msg);
    }

    /**
     * Constructs a new <code>InvalidAuthenticationException</code> with the specified detail message and cause.
     *
     * @param msg   detail message
     * @param cause the cause
     */
    public InvalidAuthenticationException(final String msg, final Throwable cause) {
        super(msg, cause);
    }

    /**
     * Constructs a new <code>InvalidAuthenticationException</code> with the specified cause.
     *
     * @param cause the cause
     */
    public InvalidAuthenticationException(final Throwable cause) {
        super(cause);
    }

    /**
     * Creates a new instance of an <code>InvalidAuthenticationException</code> with a default detail message using the
     * name of the entity that failed to authenticate.
     *
     * @param name name of entity
     * @return new instance of <code>InvalidAuthenticationException</code>
     */
    public static InvalidAuthenticationException newInstanceWithName(final String name) {
        return new InvalidAuthenticationException("Account with name <" + name + "> failed to authenticate");
    }

    /**
     * Creates a new instance of an <code>InvalidAuthenticationException</code> with a default detail message using the
     * name of the entity that failed to authenticate, and a cause.
     *
     * @param name  name of entity
     * @param cause the cause
     * @return new instance of <code>InvalidAuthenticationException</code>
     */
    public static InvalidAuthenticationException newInstanceWithName(final String name, final Throwable cause) {
        return new InvalidAuthenticationException("Account with name <" + name + "> failed to authenticate", cause);
    }

    public static InvalidAuthenticationException newInstanceWithNameAndDescriptionFromCause(final String name, final Throwable cause) {
        return new InvalidAuthenticationException("Account with name <" + name + "> failed to authenticate: "
                + StringEscapeUtils.escapeJava(cause.getMessage()));
    }

    /**
     * @since 4.4.0
     */
    @Nullable
    public String getUsername() {
        return username;
    }

    /**
     * @since 4.4.0
     */
    @Nullable
    public Directory getDirectory() {
        return directory;
    }
}
