package com.atlassian.crowd.audit;

/**
 * Describes the possible auditable event types. Historically the types were rather generic, with the entity describing
 * the target and the event describing the action that happened to it. Since Crowd 3.2.0 however the actions are verbose
 * and cover both the target object and what happened to it.
 *
 * Sample shapes of events are defined here, while Crowd will adhere to the specified contract, plugins can create events
 * with the same event types but different combinations of entities, so care must be taken when processing event data.
 */
public enum AuditLogEventType {
    /**
     * Indicates that a new entity was created during the auditable action
     * @deprecated This event type has been deprecated in favor of more specific event types (such as USER_CREATED).
     * Please use a more specific event type or {@link AuditLogEventType#OTHER} if none apply to your event.
     */
    @Deprecated
    CREATED,

    /**
     * Indicates that an entity was modified during the auditable action
     * @deprecated This event type has been deprecated in favor of more specific event types (such as USER_CREATED).
     * Please use a more specific event type or {@link AuditLogEventType#OTHER} if none apply to your event.
     */
    @Deprecated
    MODIFIED,

    /**
     * Indicates that an entity was deleted during the auditable action
     * @deprecated This event type has been deprecated in favor of more specific event types (such as USER_CREATED).
     * Please use a more specific event type or {@link AuditLogEventType#OTHER} if none apply to your event.
     */
    @Deprecated
    DELETED,

    /**
     * Indicates a start of a long running process or task
     * @deprecated This event type has been deprecated in favor of more specific event types (such as USER_CREATED).
     * Please use a more specific event type or {@link AuditLogEventType#OTHER} if none apply to your event.
     */
    @Deprecated
    STARTED,

    /**
     * Indicates a successful completion of a long running process or task
     * @deprecated This event type has been deprecated in favor of more specific event types (such as USER_CREATED).
     * Please use a more specific event type or {@link AuditLogEventType#OTHER} if none apply to your event.
     */
    @Deprecated
    COMPLETED,

    /**
     * Indicates a failure of a long running process or task
     * @deprecated This event type has been deprecated in favor of more specific event types (such as USER_CREATED).
     * Please use a more specific event type or {@link AuditLogEventType#OTHER} if none apply to your event.
     */
    @Deprecated
    FAILED,

    /**
     * Indicates that the auditable action was none of the above
     */
    OTHER,

    /**
     * Indicates the creation of a new application.
     */
    APPLICATION_CREATED,

    /**
     * Indicates the that an application's configuration was changed.
     */
    APPLICATION_UPDATED,

    /**
     * Indicates that an application was removed.
     */
    APPLICATION_DELETED,

    /**
     * Indicates the creation of a new directory.
     */
    DIRECTORY_CREATED,

    /**
     * Indicates the that a directory's configuration was changed.
     */
    DIRECTORY_UPDATED,

    /**
     * Indicates that a directory was removed.
     */
    DIRECTORY_DELETED,

    /**
     * Indicates that Crowd's configuration has been modified.
     */
    CONFIGURATION_MODIFIED,

    /**
     * Indicates that restoring Crowd from a backup file has been started.
     */
    RESTORE_STARTED,

    /**
     * Indicates that restoring Crowd from a backup file has been finished.
     */
    RESTORE_FINISHED,

    /**
     * Indicates the creation of a new user.
     */
    USER_CREATED,

    /**
     * Indicates that user data was changed.
     */
    USER_UPDATED,

    /**
     * Indicates that a user was removed.
     */
    USER_DELETED,

    /**
     * Indicates the creation of a new group.
     */
    GROUP_CREATED,

    /**
     * Indicates that group data was changed. Membership changes
     * are not covered by this event type.
     */
    GROUP_UPDATED,

    /**
     * Indicates that a group was removed.
     */
    GROUP_DELETED,

    /**
     * Indicates that a user or a group (child) has joined a group (parent). The child entity is annotated as the primary
     * entity in this event.
     */
    ADDED_TO_GROUP,

    /**
     * Indicates that a user or a group (child) has left a group (parent). The child entity is annotated as the primary
     * entity in this event.
     */
    REMOVED_FROM_GROUP,

    /**
     * Indicates that an alias was created for a given user. The participating entities are the user, for whom the alias
     * was created, and the application, in which the given alias is effective.
     */
    ALIAS_CREATED,

    /**
     * Indicates that an existing alias was changed for a given user. The participating entities are the user, for whom
     * the alias was updated, and the application, in which the given alias is effective.
     */
    ALIAS_UPDATED,

    /**
     * Indicates that an alias was deleted for a given user. The participating entities are the user, for whom the alias
     * was deleter, and the application, in which the given alias is effective.
     */
    ALIAS_DELETED,

    /**
     * Indicates that a synchronization has been started. The synchronized directory is the entity and the author will
     * either be empty (if the synchronization was scheduled) or will contain the user who triggered the synchronization.
     */
    SYNCHRONIZATION_STARTED,

    /**
     * Indicates that a synchronization has been finished. The synchronized directory is the entity and the author will
     * either be empty (if the synchronization was scheduled) or will contain the user who triggered the synchronization.
     */
    SYNCHRONIZATION_FINISHED,

    /**
     * Indicates that a directory import has been started. The target directory is the entity and the author will
     * contain the user who started the import.
     */
    IMPORT_STARTED,

    /**
     * Indicates that a directory import has been finished. The target directory is the entity and the author will
     * contain the user who started the import.
     */
    IMPORT_FINISHED,

    /**
     * Indicates that a password reset flow has been initiated. The entity is the user for whom the reset was requested
     * and the author is the user who requested the reset.
     */
    PASSWORD_RESET_STARTED,

    /**
     * Indicates that a password reset flow has been finished. The entity is the user for whom the reset was requested
     * and the author is the user who requested the reset.
     */
    PASSWORD_RESET_FINISHED,

    /**
     * Indicates that password was successfully changed for a user. The entity is the user for whom the password was
     * changed, and the author is the user who changed it.
     */
    PASSWORD_CHANGED,

    /**
     * Indicates that either a group or a user have been assigned as administrators of some other group.
     *
     * @since 3.3.0
     */
    GROUP_ADMIN_ASSIGNED,

    /**
     * Indicates that either a group or a user have been revoked administration access to some other group.
     *
     * @since 3.3.0
     */
    GROUP_ADMIN_REVOKED
}
