package com.atlassian.crowd.model.group;

import com.atlassian.crowd.attribute.AttributeUtil;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.SetMultimap;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Set;

public final class ImmutableGroupWithAttributes extends BaseImmutableGroup implements GroupWithAttributes {
    private final ImmutableSetMultimap<String, String> attributes;

    private ImmutableGroupWithAttributes(Builder builder) {
        super(builder);
        this.attributes = builder.attributes;
    }

    public static ImmutableGroupWithAttributes from(GroupWithAttributes group) {
        if (group instanceof ImmutableGroupWithAttributes) {
            return (ImmutableGroupWithAttributes) group;
        }
        return builder(group).build();
    }

    public static Builder builder(GroupWithAttributes group) {
        return builder(group, AttributeUtil.toMultimap(group));
    }

    public static Builder builder(Group group, SetMultimap<String, String> attributes) {
        return new Builder(group, attributes);
    }

    @Nonnull
    @Override
    public Set<String> getValues(String key) {
        return attributes.get(key);
    }

    @Nullable
    @Override
    public String getValue(String key) {
        return Iterables.getFirst(getValues(key), null);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    @Override
    public ImmutableGroupWithAttributes withName(String name) {
        return builder(this).setName(name).build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        ImmutableGroupWithAttributes that = (ImmutableGroupWithAttributes) o;
        return Objects.equals(attributes, that.attributes);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), attributes);
    }

    public static class Builder extends BaseImmutableGroup.Builder<Builder> {
        private ImmutableSetMultimap<String, String> attributes;

        public Builder(Group group, SetMultimap<String, String> attributes) {
            super(group);
            setAttributes(attributes);
        }

        public Builder setAttributes(SetMultimap<String, String> attributes) {
            this.attributes = ImmutableSetMultimap.copyOf(attributes);
            return this;
        }

        @Override
        public ImmutableGroupWithAttributes build() {
            return new ImmutableGroupWithAttributes(this);
        }
    }
}
