package com.atlassian.crowd.model.group;

import com.google.common.base.MoreObjects;

import java.util.Date;
import java.util.Objects;

public class ImmutableDirectoryGroup extends BaseImmutableGroup implements InternalDirectoryGroup {
    private final Date createdDate;
    private final Date updatedDate;
    private final boolean isLocal;

    private ImmutableDirectoryGroup(Builder builder) {
        super(builder);
        this.createdDate = copy(builder.createdDate);
        this.updatedDate = copy(builder.updatedDate);
        this.isLocal = builder.isLocal;
    }

    public static ImmutableDirectoryGroup from(InternalDirectoryGroup group) {
        if (group instanceof ImmutableDirectoryGroup) {
            return (ImmutableDirectoryGroup) group;
        }
        return builder(group).build();
    }

    @Override
    public ImmutableDirectoryGroup withName(String name) {
        return builder(this).setName(name).build();
    }

    public static Builder builder(InternalDirectoryGroup group) {
        return new Builder(group);
    }

    public static ImmutableDirectoryGroup.Builder builder(String name) {
        return new ImmutableDirectoryGroup.Builder(name);
    }

    @Override
    public Date getCreatedDate() {
        return copy(createdDate);
    }

    @Override
    public Date getUpdatedDate() {
        return copy(updatedDate);
    }

    @Override
    public boolean isLocal() {
        return isLocal;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        ImmutableDirectoryGroup that = (ImmutableDirectoryGroup) o;
        return Objects.equals(createdDate, that.createdDate) &&
                Objects.equals(updatedDate, that.updatedDate) &&
                isLocal == that.isLocal;
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), createdDate, updatedDate, isLocal);
    }

    public static class Builder extends BaseImmutableGroup.Builder<Builder> {
        private Date createdDate;
        private Date updatedDate;
        private boolean isLocal;

        private Builder(InternalDirectoryGroup group) {
            super(group);
            setCreatedDate(group.getCreatedDate());
            setUpdatedDate(group.getUpdatedDate());
            setLocal(group.isLocal());
        }

        private Builder(String name) {
            super(name);
        }

        public Builder setCreatedDate(Date createdDate) {
            this.createdDate = createdDate;
            return this;
        }

        public Builder setUpdatedDate(Date updatedDate) {
            this.updatedDate = updatedDate;
            return this;
        }

        public Builder setLocal(boolean local) {
            isLocal = local;
            return this;
        }

        @Override
        public ImmutableDirectoryGroup build() {
            return new ImmutableDirectoryGroup(this);
        }
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("createdDate", getCreatedDate())
                .add("updatedDate", getUpdatedDate())
                .add("local", isLocal())
                .add("directoryId", getDirectoryId())
                .add("name", getName())
                .add("type", getType())
                .add("active", isActive())
                .add("description", getDescription())
                .add("externalId", getExternalId())
                .toString();
    }

    private static Date copy(Date date) {
        return date == null ? null : new Date(date.getTime());
    }
}
