package com.atlassian.crowd.model.application;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.crowd.embedded.api.Attributes;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.model.webhook.Webhook;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * An Application in Crowd. This is the top level citizen in Crowd, where an application will have an ordered set
 * of associated {@link com.atlassian.crowd.embedded.api.Directory}'s which it can access.
 */
public interface Application extends Serializable, Attributes {
    /**
     * Returns the application ID.
     *
     * @return application ID
     */
    Long getId();

    /**
     * Returns the name of the application.
     *
     * @return name of the application
     */
    String getName();

    /**
     * Returns the type of the application.
     *
     * @return application type
     */
    ApplicationType getType();

    /**
     * Returns the description of the application.
     *
     * @return description of the application
     */
    String getDescription();

    /**
     * Returns the application password.
     *
     * @return application password
     */
    PasswordCredential getCredential();

    /**
     * Returns whether the application is a permanent application and thus cannot be removed. For instance, the Crowd
     * application is a permanent application.
     *
     * @return <tt>true</tt> if the application is permanent.
     */
    boolean isPermanent();

    /**
     * Returns whether the application is active.
     *
     * @return <tt>true</tt> if the application is active.
     */
    boolean isActive();

    /**
     * Returns the attributes of the application.
     *
     * @return attributes of the application
     */
    Map<String, String> getAttributes();

    /**
     * Returns the list of directory mappings ranked by directory priority as in perspective of the application.
     *
     * @return List of directory mappings (never null).
     * @deprecated Use {@link #getApplicationDirectoryMappings()} instead. Since 2.12.
     */
    @Deprecated
    List<DirectoryMapping> getDirectoryMappings();

    /**
     * Returns the list of directory mappings ranked by directory priority as in perspective of the application.
     *
     * @return List of directory mappings (never null).
     * @since 2.12
     */
    @Nonnull
    List<ApplicationDirectoryMapping> getApplicationDirectoryMappings();

    /**
     * Returns a directory mapping of the directory specified by directory id.
     *
     * @param directoryId ID of the directory
     * @return directory mapping if found, null if the directory mapping could not be found
     * @deprecated Use {@link #getApplicationDirectoryMapping(long)} instead. Since 2.12.
     */
    @Deprecated
    DirectoryMapping getDirectoryMapping(long directoryId);

    /**
     * Returns a directory mapping of the directory specified by directory id.
     *
     * @param directoryId ID of the directory
     * @return directory mapping if found, null if the directory mapping could not be found
     * @since 2.12
     */
    @Nullable
    ApplicationDirectoryMapping getApplicationDirectoryMapping(long directoryId);

    /**
     * Returns the whitelist of addresses allowed to connect to Crowd as the application. The remote addresses may
     * contain subnet masking information in CIDR format.
     *
     * @return whitelist of addresses allowed to connect to Crowd as the application.
     */
    Set<RemoteAddress> getRemoteAddresses();

    /**
     * Returns <tt>true</tt> if the remote address is already in the list of allowable remote addresses for the
     * application.
     *
     * @param remoteAddress RemoteAddress whose presence is to be tested
     * @return <tt>true</tt> if the remote address is already in the list of allowable remote addresses for the
     * application
     */
    boolean hasRemoteAddress(String remoteAddress);

    /**
     * Returns the Webhooks associated to the application. Webhooks were added in Crowd 2.7. For applications that
     * are not Crowd servers, it is OK to always return an empty Set.
     *
     * @return Webhooks associated to the application.
     * @since 2.7
     */
    Set<Webhook> getWebhooks();

    /**
     * @return <tt>true</tt> if user queries should return only users with access.
     */
    @ExperimentalApi
    boolean isFilteringUsersWithAccessEnabled();

    /**
     * @return <tt>true</tt> if group queries should return only groups with access.
     */
    @ExperimentalApi
    boolean isFilteringGroupsWithAccessEnabled();

    /**
     * Returns <tt>true</tt> if the usernames and group names returned should be in lowercase.
     *
     * @return <tt>true</tt> if the usernames and group names returned
     */
    boolean isLowerCaseOutput();

    /**
     * Returns <tt>true</tt> if aliasing is enabled for the application.
     *
     * @return <tt>true</tt> if aliasing is enabled for the application
     */
    boolean isAliasingEnabled();

    /**
     * @return <tt>true</tt> if membership aggregation is enabled for this application.
     */
    boolean isMembershipAggregationEnabled();

    /**
     * @return <tt>true</tt> if authentication order optimization for cached directories is enabled for this application.
     */
    boolean isCachedDirectoriesAuthenticationOrderOptimisationEnabled();

    /**
     * Returns <tt>true</tt> if application has authentication without user's password enabled, false otherwise.
     * <p>
     * Authentication without user's password is used i.e. for remember me functionality in Atlassian products.
     * If a user session is expired but remember-me cookie is still valid, product will request Crowd to generate a new
     * Crowd SSO token without passing user's password to Crowd.
     * <p>
     * As possibility to authenticate user without knowing their password is considered insecure, this option is
     * disabled by default and it should be enabled only for trusted apps.
     *
     * @return <tt>true</tt> if application has authentication without user's password enabled, false otherwise.
     */
    boolean isAuthenticationWithoutPasswordEnabled();

    /**
     * Returns the date the application was created.
     *
     * @return date the application was created
     */
    Date getCreatedDate();

    /**
     * Returns the date the application was last updated. If the application has just been created, the updated date
     * will be the same as the creation date.
     *
     * @return date the application was last updated
     */
    Date getUpdatedDate();
}
