package com.atlassian.crowd.event;

import java.util.List;

import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.event.OperationEvent;

/**
 * Represents an event store, which can be used to store events. The amount of
 * events store depends on the implementation and is hidden from the callers.
 *
 * @since 2.2
 */
public interface EventStore {
    /**
     * Returns a token that can be used for querying events that have happened
     * after the token was generated.
     * <p>
     * If the event token has not changed since the last call to this method,
     * it is guaranteed that no new events have been received.
     * <p>
     * The format of event token is implementation specific and can change
     * without a warning.
     *
     * @param directoryIds the ids of directories that the token should include events for
     * @return token that can be used for querying events that have happened
     * after the token was generated.
     */
    String getCurrentEventToken(List<Long> directoryIds);

    /**
     * Returns an events object which contains a new eventToken and events that
     * happened after the given {@code eventToken} was generated.
     * <p>
     * If for any reason event store is unable to retrieve events that happened
     * after the event token was generated, an
     * {@link EventTokenExpiredException} will be thrown. The caller is then
     * expected to call {@link #getCurrentEventToken(List)} again before asking for
     * new events.
     *
     * @param eventToken   event token that was retrieved by a call to
     *                     {@link #getCurrentEventToken(List)} or {@link #getNewEvents(String, List)}
     * @param directoryIds the ids of directories to retrieve events for
     * @return events object which contains a new eventToken and events that
     * happened after the given {@code eventToken} was generated.
     * @throws EventTokenExpiredException if events that happened after the
     *                                    event token was generated can not be retrieved
     */
    @Deprecated
    Events getNewEvents(String eventToken, List<Long> directoryIds) throws EventTokenExpiredException;

    /**
     * Returns an events object which contains a new eventToken and events that
     * happened after the given {@code eventToken} was generated.
     * <p>
     * If for any reason event store is unable to retrieve events that happened
     * after the event token was generated, an
     * {@link EventTokenExpiredException} will be thrown. The caller is then
     * expected to call {@link #getCurrentEventToken(List)} again before asking for
     * new events.
     *
     * @param eventToken   event token that was retrieved by a call to
     *                     {@link #getCurrentEventToken(List)} or {@link #getNewEvents(String, List)}
     * @param application the application to retrieve events for
     * @return events object which contains a new eventToken and events that
     * happened after the given {@code eventToken} was generated.
     * @throws EventTokenExpiredException if events that happened after the
     *                                    event token was generated can not be retrieved
     */
    Events getNewEvents(String eventToken, Application application) throws EventTokenExpiredException;

    /**
     * Stores the given event. Implementations MIGHT limit the amount
     * of events stored, and MIGHT not store the event at all if it can be recreated by other means.
     *
     * @param event event to be stored
     */
    void storeOperationEvent(OperationEvent event);

    /**
     * Notifies the store that an application event happened that might make it impossible to recreate the event
     * stream.
     *
     * Depending on the implementation the store MIGHT remove all relevant events, or mark them as invalid.
     * After the call the store MUST NOT present an event stream that might not be valid as a result of the event.
     *
     * @param event the atlassian-events event triggering the action
     */
    void handleApplicationEvent(Object event);
}
