package com.atlassian.crowd.event.group;

import java.util.Collection;
import java.util.List;
import java.util.Objects;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.event.DirectoryEvent;
import com.atlassian.crowd.model.membership.MembershipType;

import com.google.common.collect.ImmutableList;

/**
 * An Event that represents the creation of one or more Principal/Child Group to Group membership(s)
 * The event will be emitted once for each parent, and can contain a number of children added. All the children contained will
 * have the same membershipType.
 *
 * This replaces the {@link GroupMembershipCreatedEvent}. In 2.x the two events are always emitted one after the other, with the same data.
 * Listeners should listen to either {@link GroupMembershipsCreatedEvent} or {@link GroupMembershipCreatedEvent}.
 *
 * @since 2.10
 */
public class GroupMembershipsCreatedEvent extends DirectoryEvent {
    private final List<String> entityNames;
    private final String groupName;
    private final MembershipType membershipType;

    public GroupMembershipsCreatedEvent(Object source, Directory directory, Iterable<String> entityNames, String groupName, MembershipType membershipType) {
        super(source, directory);
        this.entityNames = ImmutableList.copyOf(entityNames);
        this.groupName = groupName;
        this.membershipType = membershipType;
    }

    public Collection<String> getEntityNames() {
        return entityNames;
    }

    public String getGroupName() {
        return groupName;
    }

    public MembershipType getMembershipType() {
        return membershipType;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        if (!super.equals(o)) return false;
        GroupMembershipsCreatedEvent that = (GroupMembershipsCreatedEvent) o;
        return Objects.equals(entityNames, that.entityNames) &&
                Objects.equals(groupName, that.groupName) &&
                membershipType == that.membershipType;
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), entityNames, groupName, membershipType);
    }
}
