package com.atlassian.crowd.crypto;

import com.atlassian.crowd.embedded.api.Encryptor;
import com.atlassian.crowd.exception.crypto.MissingKeyException;
import com.atlassian.crowd.manager.property.EncryptionSettings;
import com.atlassian.db.config.password.Cipher;
import com.atlassian.db.config.password.CipherProvider;
import com.atlassian.db.config.password.ciphers.algorithm.paramters.DecryptionParameters;
import com.atlassian.db.config.password.ciphers.algorithm.paramters.EncryptionParameters;
import com.google.common.annotations.VisibleForTesting;
import com.google.gson.Gson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.util.Optional;

public class DbConfigPasswordCipherEncryptor implements Encryptor {
    private static final Logger log = LoggerFactory.getLogger(DbConfigPasswordCipherEncryptor.class);
    private static final String CIPHER_PROVIDER_CLASS_NAME = "com.atlassian.db.config.password.ciphers.algorithm.AlgorithmCipher";
    private final String algorithm;
    private final String algorithmKey;
    private final EncryptionSettings encryptionSettings;

    private final ClusterAwareCipherWrapper cipher;

    @VisibleForTesting
    DbConfigPasswordCipherEncryptor(String algorithm, String algorithmKey, EncryptionSettings encryptionSettings, ClusterAwareCipherWrapper cipher) {
        this.algorithm = algorithm;
        this.algorithmKey = algorithmKey;
        this.encryptionSettings = encryptionSettings;
        this.cipher = cipher;
    }

    public DbConfigPasswordCipherEncryptor(String algorithm, String algorithmKey, EncryptionSettings encryptionSettings, CipherProvider cipherProvider) {
        this(algorithm, algorithmKey, encryptionSettings, getCipher(cipherProvider, encryptionSettings));
    }

    private static ClusterAwareCipherWrapper getCipher(CipherProvider cipherProvider, EncryptionSettings encryptionSettings) {
        Cipher cipher = cipherProvider.getInstance(CIPHER_PROVIDER_CLASS_NAME).orElseThrow(() -> new IllegalStateException(String.format("Cipher %s not found", CIPHER_PROVIDER_CLASS_NAME)));
        return new ClusterAwareCipherWrapper(encryptionSettings, cipher, new FileChecker(), new EncryptionKeyFilePermissionChanger());
    }

    @Override
    public String encrypt(String password) {
        Optional<String> encryptorDefaultKeyPath = encryptionSettings.getEncryptionKeyPath(algorithmKey);
        if (!encryptorDefaultKeyPath.isPresent()) {
            log.warn("Default encryption key is not present. Encryptor was not initialized properly.");
            throw new MissingKeyException();
        }

        DecryptionParameters decryptionParameters = doEncrypt(password, encryptorDefaultKeyPath.orElse(null));
        Gson gson = new Gson();
        return gson.toJson(decryptionParameters);
    }

    @Override
    public String decrypt(String encryptedPassword) {
        Gson gson = new Gson();
        DecryptionParameters decryptionParameters = gson.fromJson(encryptedPassword, DecryptionParameters.class);
        try {
            return cipher.decrypt(decryptionParameters);
        } catch (RuntimeException e) {
            log.error("Error during decryption", e);
            return encryptedPassword;
        }
    }

    @Override
    public boolean changeEncryptionKey() {
        DecryptionParameters decryptionParameters = doEncrypt("", null);
        encryptionSettings.setEncryptionKeyPath(algorithmKey, decryptionParameters.getKeyFilePath());
        return true;
    }

    private DecryptionParameters doEncrypt(String password, String encryptionKeyPath) {
        EncryptionParameters encryptionParameters = new EncryptionParameters.Builder()
                .setAlgorithm(algorithm)
                .setAlgorithmKey(algorithmKey)
                .setKeyFilePath(encryptionKeyPath)
                .setOutputFilesBasePath(encryptionSettings.getKeyFilesDirectoryPath() + File.separator)
                .setSaveAlgorithmParametersToSeparateFile(false)
                .setSaveSealedObjectToSeparateFile(false)
                .setPlainTextPassword(password)
                .build();

        return cipher.encrypt(encryptionParameters);
    }
}
