package com.atlassian.crowd.crypto;

import com.atlassian.crowd.exception.crypto.MissingKeyException;
import com.atlassian.crowd.manager.property.EncryptionSettings;
import com.atlassian.db.config.password.Cipher;
import com.atlassian.db.config.password.ciphers.algorithm.paramters.DecryptionParameters;
import com.atlassian.db.config.password.ciphers.algorithm.paramters.EncryptionParameters;
import com.google.gson.Gson;
import org.apache.commons.lang3.StringUtils;

/**
 * It is possible that in clustered mode Crowd nodes will have different crowd-home paths, or the path  is changed.
 * This wrapper alters encryption key path, by replacing path to key directory with a placeholder,
 * which is later substituted by each node with valid path.
 */
class ClusterAwareCipherWrapper {
    static final String KEY_DIR_PLACEHOLDER = "KEY_DIR";
    private final EncryptionSettings encryptionSettings;
    private final Cipher cipher;
    private final FileChecker fileChecker;
    private final EncryptionKeyFilePermissionChanger encryptionKeyFilePermissionChanger;

    ClusterAwareCipherWrapper(EncryptionSettings encryptionSettings, Cipher cipher, FileChecker fileChecker, EncryptionKeyFilePermissionChanger encryptionKeyFilePermissionChanger) {
        this.encryptionSettings = encryptionSettings;
        this.cipher = cipher;
        this.fileChecker = fileChecker;
        this.encryptionKeyFilePermissionChanger = encryptionKeyFilePermissionChanger;
    }

    DecryptionParameters encrypt(EncryptionParameters encryptionParameters) {
        final String keyFilePath = replacePlaceholderWithKeyDirPath(encryptionParameters.getKeyFilePath());

        if (keyFilePath != null && !fileChecker.fileExists(keyFilePath)) {
            throw new MissingKeyException(keyFilePath);
        }

        EncryptionParameters encryptionParametersWithResolvedKeyPath = new EncryptionParameters.Builder()
                .setSaveAlgorithmParametersToSeparateFile(encryptionParameters.isSaveAlgorithmParametersToSeparateFile())
                .setSaveSealedObjectToSeparateFile(encryptionParameters.isSaveSealedObjectToSeparateFile())
                .setAlgorithm(encryptionParameters.getAlgorithm())
                .setAlgorithmKey(encryptionParameters.getAlgorithmKey())
                .setPlainTextPassword(encryptionParameters.getPlainTextPassword())
                .setOutputFilesBasePath(encryptionParameters.getOutputFilesBasePath())
                .setKeyFilePath(keyFilePath)
                .build();

        Gson gson = new Gson();
        String decryptionParametersJson = cipher.encrypt(gson.toJson(encryptionParametersWithResolvedKeyPath));
        DecryptionParameters encrypt = gson.fromJson(decryptionParametersJson, DecryptionParameters.class);

        encryptionKeyFilePermissionChanger.makeFileReadableOnlyByOwner(encrypt.getKeyFilePath());

        return new DecryptionParameters.Builder()
                .serializedSealedObject(encrypt.getSerializedSealedObject())
                .setKeyFilePath(replaceKeyDirPathWithPlaceholder(encrypt.getKeyFilePath()))
                .setSealedObjectFilePath(encrypt.getSealedObjectFilePath())
                .build();
    }

    String decrypt(DecryptionParameters decryptionParameters) {
        DecryptionParameters decryptionParametersWithPlaceholderInPath = new DecryptionParameters.Builder()
                .serializedSealedObject(decryptionParameters.getSerializedSealedObject())
                .setKeyFilePath(replacePlaceholderWithKeyDirPath(decryptionParameters.getKeyFilePath()))
                .setSealedObjectFilePath(decryptionParameters.getSealedObjectFilePath())
                .build();

        Gson gson = new Gson();
        return cipher.decrypt(gson.toJson(decryptionParametersWithPlaceholderInPath));
    }

    private String replaceKeyDirPathWithPlaceholder(String path) {
        return StringUtils.replace(path, encryptionSettings.getKeyFilesDirectoryPath(), KEY_DIR_PLACEHOLDER);
    }

    private String replacePlaceholderWithKeyDirPath(String path) {
        return StringUtils.replace(path, KEY_DIR_PLACEHOLDER, encryptionSettings.getKeyFilesDirectoryPath());
    }
}
