package com.atlassian.crowd.search.util;

import com.atlassian.crowd.search.query.entity.EntityQuery;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.function.Predicate;

/**
 * Helper class constraining results according to the start index and maximum results.
 */
public class OrderedResultsConstrainer<T> {
    private final List<T> results = new ArrayList<>();
    private final Predicate<T> filter;
    private final int startIndex;
    private final int endIndex;
    private int currentIndex;

    /**
     * @param filter results filter, {@code null} means no filtering
     * @param startIndex results with index lower than {@code startIndex} will be dropped
     * @param maxResults maximum number of results, for all results specify {@link EntityQuery#ALL_RESULTS}
     */
    public OrderedResultsConstrainer(Predicate<T> filter, int startIndex, int maxResults) {
        this.filter = filter;
        this.startIndex = startIndex;
        this.endIndex = EntityQuery.addToMaxResults(maxResults, startIndex);
        this.currentIndex = 0;
    }

    /**
     * @param collection collection to append to results
     */
    public void addAll(Iterable<T> collection) {
        for (Iterator<T> it = collection.iterator(); getRemainingCount() != 0 && it.hasNext(); ) {
            T next = it.next();
            if (filter == null || filter.test(next)) {
                if (currentIndex++ >= startIndex) {
                    results.add(next);
                }
            }
        }
    }

    /**
     * Returns constrained results.
     */
    public List<T> toList() {
        return results;
    }

    /**
     * Returns remaining number of results that need to be fetched. If {@link EntityQuery#ALL_RESULTS} were requested
     * then returns {@link EntityQuery#ALL_RESULTS} .
     */
    public int getRemainingCount() {
        return isAllResultsQuery() ? EntityQuery.ALL_RESULTS : (endIndex - currentIndex);
    }

    private boolean isAllResultsQuery() {
        return endIndex == EntityQuery.ALL_RESULTS;
    }
}
