package com.atlassian.crowd.manager.recovery;

import com.atlassian.crowd.common.properties.StringSystemProperty;
import com.atlassian.crowd.directory.loader.DirectoryInstanceLoader;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.event.recovery.RecoveryModeActivatedEvent;
import com.atlassian.event.api.EventPublisher;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * An implementation of {@link RecoveryModeService} that activates recovery mode in presence of a system property that
 * is also used to configure the recovery user password.
 *
 * @since 2.7.2
 */
public class SystemPropertyRecoveryModeService implements RecoveryModeService {
    private static final Logger logger = LoggerFactory.getLogger(SystemPropertyRecoveryModeService.class);

    static final String RECOVERY_USERNAME = "recovery_admin";
    static final String RECOVERY_DISPLAY_NAME = "Recovery Admin User";
    static final String RECOVERY_EMAIL = "@";
    @VisibleForTesting
    public static final StringSystemProperty PROP_RECOVERY_PASSWORD = new StringSystemProperty("atlassian.recovery.password", "");

    private final DirectoryInstanceLoader loader;
    private final EventPublisher eventPublisher;
    private final AtomicBoolean recoveryModeEventRaised = new AtomicBoolean();

    private final Supplier<String> recoveryPasswordRef = Suppliers.memoize(PROP_RECOVERY_PASSWORD::getValue);

    private final Supplier<RecoveryModeDirectory> recoveryDirectoryRef = Suppliers.memoize(
            () -> new RecoveryModeDirectory(RECOVERY_USERNAME, recoveryPasswordRef.get()));

    public SystemPropertyRecoveryModeService(DirectoryInstanceLoader loader, EventPublisher eventPublisher) {
        this.loader = checkNotNull(loader, "loader");
        this.eventPublisher = checkNotNull(eventPublisher, "eventPublisher");
    }

    @Override
    public boolean isRecoveryModeOn() {
        final boolean recoveryModeOn = isRecoveryFeatureOn() && isNotBlank(getRecoveryPassword());
        if (recoveryModeEventRaised.compareAndSet(false, true)) {
            if (recoveryModeOn) {
                // if on, notify of the recovery mode on first access
                logger.info("Recovery mode is ON");
                eventPublisher.publish(new RecoveryModeActivatedEvent(RECOVERY_USERNAME, getRecoveryDirectory()));
            } else if (isNotBlank(getRecoveryPassword()) && !isRecoveryFeatureOn()) {
                // someone activated the recovery mode, but the feature is not supported by the host application
                logger.warn("Attempt has been made to activate recovery mode, but the host application does not support it yet");
            }
        }
        return recoveryModeOn;
    }

    @Override
    public Directory getRecoveryDirectory() {
        Preconditions.checkState(isRecoveryModeOn(), "Recovery Mode is not ON");
        return recoveryDirectoryRef.get();
    }

    @Override
    public String getRecoveryUsername() {
        Preconditions.checkState(isRecoveryModeOn(), "Recovery Mode is not ON");
        return RECOVERY_USERNAME;
    }

    @Override
    public boolean isRecoveryDirectory(final Directory directory) {
        return directory.getId().equals(RecoveryModeDirectory.ID);
    }

    private boolean isRecoveryFeatureOn() {
        // we need to be able to load the RecoveryModeDirectory for the recovery mode to work.
        // this basically means that RecoveryModeDirectoryLoader is included in the directory loader chain
        return loader.canLoad(RecoveryModeDirectory.class.getName());
    }

    private String getRecoveryPassword() {
        return recoveryPasswordRef.get();
    }
}
