package com.atlassian.crowd.manager.recovery;

import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.core.event.MultiEventPublisher;
import com.atlassian.crowd.dao.application.ApplicationDAO;
import com.atlassian.crowd.directory.loader.DirectoryInstanceLoader;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.manager.directory.BeforeGroupRemoval;
import com.atlassian.crowd.manager.directory.DirectoryManagerGeneric;
import com.atlassian.crowd.manager.directory.DirectorySynchroniser;
import com.atlassian.crowd.manager.directory.SynchronisationStatusManager;
import com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerManager;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsCacheProvider;
import com.atlassian.crowd.manager.permission.PermissionManager;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Optional;

import static com.atlassian.crowd.search.query.DirectoryQueries.allDirectories;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Overrides {@link DirectoryManagerGeneric} to provide recovery-mode aware behaviour.
 *
 * @since 2.7.2
 */
public class RecoveryModeAwareDirectoryManager extends DirectoryManagerGeneric {
    private final RecoveryModeService recoveryModeService;

    public RecoveryModeAwareDirectoryManager(DirectoryDao directoryDao, ApplicationDAO applicationDAO,
                                             MultiEventPublisher eventPublisher, PermissionManager permissionManager,
                                             DirectoryInstanceLoader directoryInstanceLoader,
                                             DirectorySynchroniser directorySynchroniser,
                                             DirectoryPollerManager directoryPollerManager,
                                             ClusterLockService clusterLockService,
                                             SynchronisationStatusManager synchronisationStatusManager,
                                             BeforeGroupRemoval beforeGroupRemoval,
                                             RecoveryModeService recoveryModeService,
                                             Optional<NestedGroupsCacheProvider> nestedGroupsCacheProvider) {
        super(directoryDao, applicationDAO, eventPublisher, permissionManager, directoryInstanceLoader,
                directorySynchroniser, directoryPollerManager, clusterLockService, synchronisationStatusManager,
                beforeGroupRemoval, nestedGroupsCacheProvider);
        this.recoveryModeService = checkNotNull(recoveryModeService, "recoveryModeService");
    }

    @Override
    public Directory findDirectoryById(long directoryId) throws DirectoryNotFoundException {
        if (recoveryModeService.isRecoveryModeOn() && recoveryModeService.getRecoveryDirectory().getId().equals(directoryId)) {
            return recoveryModeService.getRecoveryDirectory();
        } else {
            return super.findDirectoryById(directoryId);
        }
    }

    @Override
    public List<Directory> findAllDirectories() {
        final List<Directory> directories = super.findAllDirectories();
        return addRecoveryDirectoryIfNeeded(directories);
    }

    @Override
    public List<Directory> searchDirectories(final EntityQuery<Directory> query) {
        final List<Directory> directories = super.searchDirectories(query);
        if (query.equals(allDirectories())) {
            return addRecoveryDirectoryIfNeeded(directories);
        } else {
            return directories;
        }
    }

    private List<Directory> addRecoveryDirectoryIfNeeded(final List<Directory> directories) {
        if (recoveryModeService.isRecoveryModeOn()) {
            return ImmutableList.<Directory>builder()
                    .add(recoveryModeService.getRecoveryDirectory())
                    .addAll(directories)
                    .build();
        } else {
            return directories;
        }
    }

    @Override
    public Directory findDirectoryByName(String name) throws DirectoryNotFoundException {
        if (recoveryModeService.isRecoveryModeOn() && recoveryModeService.getRecoveryDirectory().getName().equalsIgnoreCase(name)) {
            return recoveryModeService.getRecoveryDirectory();
        } else {
            return super.findDirectoryByName(name);
        }
    }

}
