package com.atlassian.crowd.manager.property;

import com.atlassian.crowd.dao.property.PropertyDAO;
import com.atlassian.crowd.event.configuration.AuditLogConfigurationUpdatedEvent;
import com.atlassian.crowd.event.configuration.ConfigurationPropertyUpdatedEvent;
import com.atlassian.crowd.event.configuration.LookAndFeelUpdatedEvent;
import com.atlassian.crowd.event.configuration.SmtpServerUpdatedEvent;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.integration.Constants;
import com.atlassian.crowd.manager.audit.AuditLogConfiguration;
import com.atlassian.crowd.manager.audit.RetentionPeriod;
import com.atlassian.crowd.manager.authentication.ImmutableCrowdSpecificRememberMeSettings;
import com.atlassian.crowd.manager.rememberme.CrowdSpecificRememberMeSettings;
import com.atlassian.crowd.model.authentication.CookieConfiguration;
import com.atlassian.crowd.model.backup.BackupConfiguration;
import com.atlassian.crowd.model.lookandfeel.LookAndFeelConfiguration;
import com.atlassian.crowd.model.property.Property;
import com.atlassian.crowd.password.encoder.DESPasswordEncoder;
import com.atlassian.crowd.util.ImageInfo;
import com.atlassian.crowd.util.mail.SMTPServer;
import com.atlassian.event.api.EventPublisher;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.primitives.Longs;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;
import org.codehaus.jackson.map.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.DESKeySpec;
import javax.mail.internet.InternetAddress;
import java.io.IOException;
import java.net.URI;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import static com.atlassian.crowd.model.property.Property.AUDIT_LOG_RETENTION_PERIOD;
import static com.atlassian.crowd.model.property.Property.BACKUP_SCHEDULED_TIME_HOUR;
import static com.atlassian.crowd.model.property.Property.BACKUP_SCHEDULED_TIME_MINUTE;
import static com.atlassian.crowd.model.property.Property.BUILD_NUMBER;
import static com.atlassian.crowd.model.property.Property.CACHE_ENABLED;
import static com.atlassian.crowd.model.property.Property.CROWD_BASE_URL;
import static com.atlassian.crowd.model.property.Property.CROWD_PROPERTY_KEY;
import static com.atlassian.crowd.model.property.Property.CURRENT_LICENSE_RESOURCE_TOTAL;
import static com.atlassian.crowd.model.property.Property.DATABASE_TOKEN_STORAGE_ENABLED;
import static com.atlassian.crowd.model.property.Property.DEPLOYMENT_TITLE;
import static com.atlassian.crowd.model.property.Property.DES_ENCRYPTION_KEY;
import static com.atlassian.crowd.model.property.Property.DOMAIN;
import static com.atlassian.crowd.model.property.Property.EXPORT_USERS_FROM_CONNECTORS_DURING_BACKUP_ENABLED;
import static com.atlassian.crowd.model.property.Property.FORGOTTEN_PASSWORD_EMAIL_TEMPLATE;
import static com.atlassian.crowd.model.property.Property.INCLUDE_IP_ADDRESS_IN_VALIDATION_FACTORS;
import static com.atlassian.crowd.model.property.Property.LOOK_AND_FEEL_CONFIGURATION_PROPERTY_NAME;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_HOST;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_JNDI_LOCATION;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_PASSWORD;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_PORT;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_PREFIX;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_SENDER;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_START_TLS;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_TIMEOUT;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_USERNAME;
import static com.atlassian.crowd.model.property.Property.MAILSERVER_USE_SSL;
import static com.atlassian.crowd.model.property.Property.NOTIFICATION_EMAIL;
import static com.atlassian.crowd.model.property.Property.REMEMBER_ME_ENABLED_PROPERTY_NAME;
import static com.atlassian.crowd.model.property.Property.REMEMBER_ME_EXPIRY_IN_SECONDS_PROPERTY_NAME;
import static com.atlassian.crowd.model.property.Property.RESET_DOMAIN_FOR_BACKUP_ENABLED;
import static com.atlassian.crowd.model.property.Property.SAML_KEY_CERTIFICATE_PAIR_TO_SIGN;
import static com.atlassian.crowd.model.property.Property.SCHEDULED_BACKUP_ENABLED;
import static com.atlassian.crowd.model.property.Property.SECURE_COOKIE;
import static com.atlassian.crowd.model.property.Property.SESSION_TIME;
import static com.atlassian.crowd.model.property.Property.SSO_COOKE_NAME_PROPERTY;
import static com.atlassian.crowd.model.property.Property.TRUSTED_PROXY_SERVERS;
import static com.atlassian.crowd.model.property.Property.USE_WEB_AVATARS;
import static com.google.common.base.Strings.emptyToNull;

@Transactional
public class PropertyManagerGeneric implements InternalPropertyManager {
    private static final Logger logger = LoggerFactory.getLogger(PropertyManagerGeneric.class);
    private static final int DEFAULT_SESSION_TIME_IN_MINUTES = 5;
    public static final int DEFAULT_SCHEDULED_BACKUP_HOUR = 2;
    public static final int DEFAULT_SCHEDULED_BACKUP_MINUTE = 0;
    private static final ObjectMapper JSON_MAPPER = new ObjectMapper();

    public static final String LOGO_IMAGE_KEY = "logo_image_info_property_key";
    static final String BASE64_FILE_PROPERTY_NAME_FORMAT = "file.%s";

    private final PropertyDAO propertyDAO;
    private final EventPublisher eventPublisher;

    public PropertyManagerGeneric(PropertyDAO propertyDAO, EventPublisher eventPublisher) {
        this.propertyDAO = propertyDAO;
        this.eventPublisher = eventPublisher;
    }

    @Override
    public String getDeploymentTitle() throws PropertyManagerException {
        return getPropertyInternal(DEPLOYMENT_TITLE);
    }

    @Override
    public void setDeploymentTitle(String title) {
        setProperty(DEPLOYMENT_TITLE, title);
    }

    @Override
    public String getDomain() {
        return getString(DOMAIN, null);
    }

    @Override
    public void setDomain(String domain) {
        setProperty(DOMAIN, domain);
    }

    @Override
    public boolean isSecureCookie() {
        return getBoolean(SECURE_COOKIE, false);
    }

    @Override
    public void setSecureCookie(boolean secure) {
        setBooleanProperty(SECURE_COOKIE, isSecureCookie(), secure);
    }

    @Override
    public void setCacheEnabled(boolean enabled) {
        setBooleanProperty(CACHE_ENABLED, isCacheEnabled(), enabled);
    }

    @Override
    public boolean isCacheEnabled() {
        return getBoolean(CACHE_ENABLED, false);
    }

    @Override
    public long getSessionTime() {
        return getOptionalProperty(SESSION_TIME)
                .map(Longs::tryParse)
                .map(TimeUnit.MILLISECONDS::toMinutes)
                .orElse((long)DEFAULT_SESSION_TIME_IN_MINUTES);
    }

    @Override
    public void setSessionTime(long time) {
        //comes in as minutes, store in milliseconds
        setProperty(SESSION_TIME, Long.toString(TimeUnit.MINUTES.toMillis(time)));
    }

    @Override
    public SMTPServer getSMTPServer() throws PropertyManagerException {
        // Required Field
        InternetAddress fromAddress = getPropertyInternal(MAILSERVER_SENDER, InternetAddress::new);
        String prefix = getString(MAILSERVER_PREFIX, null);

        String jndiLocation = getString(MAILSERVER_JNDI_LOCATION, null);
        return Strings.isNullOrEmpty(jndiLocation) ? buildSMTPServer(fromAddress, prefix)
                : new SMTPServer(jndiLocation, fromAddress, prefix);
    }

    private SMTPServer buildSMTPServer(InternetAddress fromAddress, String prefix)
            throws PropertyManagerException {
        String host = getPropertyInternal(MAILSERVER_HOST);

        // Optional Attributes
        String password = getString(MAILSERVER_PASSWORD, null);
        String username = getString(MAILSERVER_USERNAME, null);

        int port = getIntOrThrowIllegalArgumentException(MAILSERVER_PORT, SMTPServer.DEFAULT_MAIL_PORT);
        boolean useSSL = getBoolean(MAILSERVER_USE_SSL, false);
        int timeout = getIntOrThrowIllegalArgumentException(MAILSERVER_TIMEOUT, SMTPServer.DEFAULT_TIMEOUT);
        boolean startTLS = getBoolean(MAILSERVER_START_TLS, false);

        return SMTPServer.builder()
                .setPort(port)
                .setPrefix(prefix)
                .setFrom(fromAddress)
                .setPassword(password)
                .setUsername(username)
                .setHost(host)
                .setUseSSL(useSSL)
                .setTimeout(timeout)
                .setStartTLS(startTLS)
                .build();
    }

    @Override
    public void setSMTPServer(SMTPServer server) {
        final SMTPServer oldValue = safeGetSMTPServer();
        setProperty(MAILSERVER_PREFIX, server.getPrefix(), false);
        setProperty(MAILSERVER_SENDER, server.getFrom().toString(), false);

        if (StringUtils.isNotBlank(server.getJndiLocation())) {
            setProperty(MAILSERVER_JNDI_LOCATION, server.getJndiLocation(), false);

            // Blank out the SMTP properties
            setProperty(MAILSERVER_HOST, "", false);
            setProperty(MAILSERVER_PASSWORD, "", false);
            setProperty(MAILSERVER_USERNAME, "", false);
            setProperty(MAILSERVER_PORT, "", false);
            setProperty(MAILSERVER_USE_SSL, "", false);
            setProperty(MAILSERVER_TIMEOUT, "", false);
            setProperty(MAILSERVER_START_TLS, "", false);
        } else {
            setProperty(MAILSERVER_HOST, server.getHost(), false);
            setProperty(MAILSERVER_PASSWORD, server.getPassword(), false);
            setProperty(MAILSERVER_USERNAME, server.getUsername(), false);
            setProperty(MAILSERVER_PORT, String.valueOf(server.getPort()), false);
            setProperty(MAILSERVER_USE_SSL, String.valueOf(server.getUseSSL()), false);
            setProperty(MAILSERVER_TIMEOUT, String.valueOf(server.getTimeout()), false);
            setProperty(MAILSERVER_START_TLS, String.valueOf(server.isStartTLS()) , false);

            // Blank out the JNDI server location
            setProperty(MAILSERVER_JNDI_LOCATION, "", false);
        }
        final SMTPServer newValue = safeGetSMTPServer();
        if (!Objects.equals(oldValue, newValue)) {
            eventPublisher.publish(new SmtpServerUpdatedEvent(oldValue, newValue));
        }
    }

    private SMTPServer safeGetSMTPServer() {
        try {
            return getSMTPServer();
        } catch (PropertyManagerException e) {
            return null;
        }
    }

    @Override
    public Key getDesEncryptionKey() throws PropertyManagerException {
        return getPropertyInternal(DES_ENCRYPTION_KEY, keyStr -> {
            // create a DES key spec
            DESKeySpec ks = new DESKeySpec(Base64.decodeBase64(keyStr));

            // generate the key from the DES key spec
            return SecretKeyFactory.getInstance(DESPasswordEncoder.PASSWORD_ENCRYPTION_ALGORITHM).generateSecret(ks);
        });
    }

    @Override
    @SuppressFBWarnings(value = "DES_USAGE", justification = "Only used for DESPasswordEncoder, which is not used by default")
    public void generateDesEncryptionKey() throws PropertyManagerException {
        if (getOptionalProperty(DES_ENCRYPTION_KEY).isPresent()) {
            // only generate the key if it does not already exist
            return;
        }
        try {
            // create a new key
            Key key = KeyGenerator.getInstance(DESPasswordEncoder.PASSWORD_ENCRYPTION_ALGORITHM).generateKey();

            // store this key
            setProperty(DES_ENCRYPTION_KEY, Base64.encodeBase64String(key.getEncoded()));

        } catch (NoSuchAlgorithmException e) {
            throw new PropertyManagerException(e.getMessage(), e);
        }
    }

    @Deprecated
    @Override
    public void setSMTPTemplate(String template) {
        setProperty(FORGOTTEN_PASSWORD_EMAIL_TEMPLATE, template);
    }

    @Deprecated
    @Override
    public String getSMTPTemplate() throws PropertyManagerException {
        return getPropertyInternal(FORGOTTEN_PASSWORD_EMAIL_TEMPLATE);
    }

    @Override
    public void setCurrentLicenseResourceTotal(int total) {
        setProperty(CURRENT_LICENSE_RESOURCE_TOTAL, Integer.toString(total), false);
    }

    @Override
    public int getCurrentLicenseResourceTotal() {
        try {
            return getInt(CURRENT_LICENSE_RESOURCE_TOTAL, 0);
        } catch (Exception e) {
            // The license validation happens before the upgrade to let the user go back to the previous Crowd version.
            // Because of that this function is called during startup, before the upgrade. This catch lets the Crowd
            // start if the property table is missing.
            logger.debug("Failed to find current resource total.", e);
        }
        return 0;
    }

    @Override
    public void setNotificationEmail(String notificationEmail) {
        setNotificationEmails(ImmutableList.of(notificationEmail));
    }

    @Override
    public void setNotificationEmails(List<String> serverAlertAddresses) {
        try {
            setProperty(NOTIFICATION_EMAIL, JSON_MAPPER.writeValueAsString(serverAlertAddresses));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getNotificationEmail() throws PropertyManagerException {
        try {
            final List<String> notificationEmails = getNotificationEmails();
            return notificationEmails.get(0);
        } catch (Exception e) {
            throw new PropertyManagerException(e.getMessage(), e);
        }
    }

    @Override
    public List<String> getNotificationEmails() throws PropertyManagerException {
        return getPropertyInternal(NOTIFICATION_EMAIL,
                value -> ImmutableList.copyOf((JSON_MAPPER.readValue(value, String[].class))));
    }

    @Override
    public boolean isGzipEnabled() throws PropertyManagerException {
        return true;
    }

    @Override
    public void setGzipEnabled(boolean gzip) {
        // NOP
    }

    @Override
    public Integer getBuildNumber() throws PropertyManagerException {
        return getPropertyInternal(BUILD_NUMBER, Integer::valueOf);
    }

    @Override
    public void setBuildNumber(Integer buildNumber) {
        setProperty(BUILD_NUMBER, buildNumber.toString());
    }

    /**
     * Retrieves a String that contains a list of proxy servers we trust to correctly set the X-Forwarded-For flag.
     * Internal format of this string is the responsibility of {@link com.atlassian.crowd.manager.proxy.TrustedProxyManager}.
     *
     * @return list of proxy servers we trust
     * @throws PropertyManagerException If the list of proxy servers could not be found.
     */
    @Override
    public String getTrustedProxyServers() throws PropertyManagerException {
        return getPropertyInternal(TRUSTED_PROXY_SERVERS);
    }

    /**
     * Persists a String containing a list of proxy servers we trust to correctly set the X-Forwarded-For flag.
     * Internal format of this string is the responsibility of {@link com.atlassian.crowd.manager.proxy.TrustedProxyManager}.
     *
     * @throws org.springframework.dao.DataAccessException If the list of proxy servers could not be saved.
     */
    @Override
    public void setTrustedProxyServers(String proxyServers) {
        setProperty(TRUSTED_PROXY_SERVERS, proxyServers);
    }

    @Override
    public void setAuditLogConfiguration(AuditLogConfiguration newConfiguration) {
        final AuditLogConfiguration oldConfiguration = getAuditLogConfiguration();
        if (!Objects.equals(oldConfiguration, newConfiguration)) {
            setProperty(AUDIT_LOG_RETENTION_PERIOD, newConfiguration.getRetentionPeriod().name(), false);
            eventPublisher.publish(new AuditLogConfigurationUpdatedEvent(oldConfiguration, newConfiguration));
        }
    }

    @Override
    public AuditLogConfiguration getAuditLogConfiguration() {
        try {
            return new AuditLogConfiguration(RetentionPeriod.valueOf(getProperty(AUDIT_LOG_RETENTION_PERIOD)));
        } catch (ObjectNotFoundException | IllegalArgumentException e) {
            if (e instanceof IllegalArgumentException) {
                logger.warn("Found invalid audit log retention period persisted in database, using the default instead", e);
            }
            return AuditLogConfiguration.defaultConfiguration();
        }
    }

    @Override
    public boolean isUsingDatabaseTokenStorage() {
        return getBoolean(DATABASE_TOKEN_STORAGE_ENABLED, true);
    }

    @Override
    public void setUsingDatabaseTokenStorage(boolean isUsingDatabaseTokenStorage) {
        setBooleanProperty(DATABASE_TOKEN_STORAGE_ENABLED, isUsingDatabaseTokenStorage(), isUsingDatabaseTokenStorage);
    }

    @Override
    public boolean isIncludeIpAddressInValidationFactors() {
        return getBoolean(INCLUDE_IP_ADDRESS_IN_VALIDATION_FACTORS, true);
    }

    @Override
    public boolean isUseWebAvatars() {
        return getBoolean(USE_WEB_AVATARS, false);
    }

    @Override
    public void setUseWebAvatars(boolean useWebAvatars) {
        setBooleanProperty(USE_WEB_AVATARS, isUseWebAvatars(), useWebAvatars);
    }

    @Override
    public CookieConfiguration getCookieConfiguration() {
        final String cookieName = getString(SSO_COOKE_NAME_PROPERTY, Constants.COOKIE_TOKEN_KEY);
        return new CookieConfiguration(getDomain(), isSecureCookie(), cookieName);
    }

    @Override
    public void setCookieConfiguration(CookieConfiguration cookieConfiguration) {
        setDomain(cookieConfiguration.getDomain());
        setSecureCookie(cookieConfiguration.isSecure());
        setProperty(SSO_COOKE_NAME_PROPERTY, cookieConfiguration.getName());
    }


    @Override
    public void setIncludeIpAddressInValidationFactors(boolean includeIpAddressInValidationFactors) {
        setBooleanProperty(INCLUDE_IP_ADDRESS_IN_VALIDATION_FACTORS, isIncludeIpAddressInValidationFactors(),
                includeIpAddressInValidationFactors);
    }

    @Override
    public void setRememberMeConfiguration(CrowdSpecificRememberMeSettings configuration) {
        setProperty(REMEMBER_ME_ENABLED_PROPERTY_NAME, Boolean.toString(configuration.isEnabled()));
        setProperty(REMEMBER_ME_EXPIRY_IN_SECONDS_PROPERTY_NAME, Long.toString(configuration.getExpirationDuration().getSeconds()));
    }

    @Override
    public CrowdSpecificRememberMeSettings getRememberMeConfiguration() {
        final boolean enabled = getBoolean(REMEMBER_ME_ENABLED_PROPERTY_NAME, CrowdSpecificRememberMeSettings.DEFAULT_ENABLED);
        final Duration expirationDuration = getOptionalProperty(REMEMBER_ME_EXPIRY_IN_SECONDS_PROPERTY_NAME)
                .map(Long::parseLong)
                .map(Duration::ofSeconds)
                .orElse(CrowdSpecificRememberMeSettings.DEFAULT_EXPIRATION_DURATION);
        return new ImmutableCrowdSpecificRememberMeSettings(enabled, expirationDuration);
    }

    public void removeProperty(String name) {
        propertyDAO.remove(CROWD_PROPERTY_KEY, name);
    }

    protected Property getPropertyObject(String name) throws ObjectNotFoundException {
        return propertyDAO.find(CROWD_PROPERTY_KEY, name);
    }

    void setBooleanProperty(final String name, final boolean from, final boolean to) {
        setProperty(name, Boolean.toString(to), false);
        if (from != to) {
            eventPublisher.publish(new ConfigurationPropertyUpdatedEvent(name, Boolean.toString(from), Boolean.toString(to)));
        }
    }

    @Override
    public String getProperty(String name) throws ObjectNotFoundException {
        Property property = getPropertyObject(name);
        return property.getValue();
    }

    @Override
    public Optional<String> getOptionalProperty(String name) {
        try {
            return Optional.ofNullable(getProperty(name));
        } catch (ObjectNotFoundException e) {
            return Optional.empty();
        }
    }

    private String getPropertyInternal(String name) throws PropertyManagerException {
        try {
            return getProperty(name);
        } catch (ObjectNotFoundException e) {
            throw new PropertyManagerException(e.getMessage(), e);
        }
    }

    private <T> T getPropertyInternal(String name, PropertyTransformer<T> transformer) throws PropertyManagerException {
        try {
            return transformer.apply(getProperty(name));
        } catch (Exception e) {
            Throwables.propagateIfPossible(e, PropertyManagerException.class);
            throw new PropertyManagerException(e.getMessage(), e);
        }
    }

    @Override
    public void setProperty(final String name, final String value) {
        setProperty(name, value, true);
    }

    @VisibleForTesting
    void setProperty(final String name, final String value, final boolean publishEvent) {
        Property property = null;

        try {
            property = getPropertyObject(name);
        } catch (ObjectNotFoundException e) {
            // ignore, we just want to update if the property already exist
        }

        final String oldValue;
        if (property == null) {
            property = new Property(CROWD_PROPERTY_KEY, name, value);
            oldValue = null;
        } else {
            oldValue = property.getValue();
            property.setValue(value);
        }

        // add the property to the database
        propertyDAO.update(property);
        final String oldValueOrNull = emptyToNull(oldValue);
        final String newValueOrNull = emptyToNull(value);
        if (publishEvent && !Objects.equals(oldValueOrNull, newValueOrNull)) {
            eventPublisher.publish(new ConfigurationPropertyUpdatedEvent(name, oldValueOrNull, newValueOrNull));
        }
    }

    @Override
    public String getString(String property, String defaultValue) {
        return getOptionalProperty(property).orElse(defaultValue);
    }

    @Override
    public boolean getBoolean(String property, boolean defaultValue) {
        return getOptionalProperty(property).map(Boolean::valueOf).orElse(defaultValue);
    }

    @Override
    public int getInt(String property, int defaultValue) {
        return getInt(property, defaultValue, true);
    }

    public int getIntOrThrowIllegalArgumentException(String property, int defaultValue) {
        return getInt(property, defaultValue, false);
    }

    private int getInt(String property, int defaultValue, boolean logOnly) throws IllegalArgumentException {
        Optional<String> value = getOptionalProperty(property);
        try {
            return value.map(Integer::parseInt).orElse(defaultValue);
        } catch (NumberFormatException e) {
            if (logOnly) {
                logger.warn("Corrupted value found for property {}. Found {} instead of an integer", property, value.orElse(null));
                return defaultValue;
            }
            throw new IllegalArgumentException(property + " is not a valid number", e);
        }
    }

    @Override
    public void setBaseUrl(URI url) {
        Preconditions.checkArgument(url.isAbsolute(), "Base url needs to be absolute");
        setProperty(CROWD_BASE_URL, url.toString());
    }

    @Override
    public URI getBaseUrl() throws PropertyManagerException {
        return getPropertyInternal(CROWD_BASE_URL, URI::new);
    }

    private interface PropertyTransformer<T> {
        T apply(String value) throws Exception;
    }

    @Override
    public Optional<Long> getPrivateKeyCertificatePairToSign() {
        return getOptionalProperty(SAML_KEY_CERTIFICATE_PAIR_TO_SIGN)
                .map(Long::parseLong);
    }

    @Override
    public void setPrivateKeyCertificateToSign(long privateKeyCertificatePairId) {
        setProperty(SAML_KEY_CERTIFICATE_PAIR_TO_SIGN, String.valueOf(privateKeyCertificatePairId), false);
    }

    @Override
    public BackupConfiguration getBackupConfiguration() {
        final boolean restoreUsersFromConnectors = getBoolean(EXPORT_USERS_FROM_CONNECTORS_DURING_BACKUP_ENABLED, false);
        final boolean scheduledBackupEnabled = getBoolean(SCHEDULED_BACKUP_ENABLED, true);
        final boolean resetDomainEnabled = getBoolean(RESET_DOMAIN_FOR_BACKUP_ENABLED, true);
        final int scheduledTimeHour = getInt(BACKUP_SCHEDULED_TIME_HOUR, DEFAULT_SCHEDULED_BACKUP_HOUR);
        final int scheduledTimeMinute = getInt(BACKUP_SCHEDULED_TIME_MINUTE, DEFAULT_SCHEDULED_BACKUP_MINUTE);

        return BackupConfiguration.builder()
                .setBackupConnectorEnabled(restoreUsersFromConnectors)
                .setScheduledBackupEnabled(scheduledBackupEnabled)
                .setResetDomainEnabled(resetDomainEnabled)
                .setBackupTimeHour(scheduledTimeHour)
                .setBackupTimeMinute(scheduledTimeMinute)
                .build();
    }

    @Override
    public void saveBackupConfiguration(BackupConfiguration config) {
        setProperty(EXPORT_USERS_FROM_CONNECTORS_DURING_BACKUP_ENABLED, Boolean.toString(config.isBackupConnectorEnabled()));
        setProperty(SCHEDULED_BACKUP_ENABLED, Boolean.toString(config.isScheduledBackupEnabled()));
        setProperty(RESET_DOMAIN_FOR_BACKUP_ENABLED, Boolean.toString(config.isResetDomainEnabled()));
        setProperty(BACKUP_SCHEDULED_TIME_HOUR, String.valueOf(config.getBackupTimeHour()));
        setProperty(BACKUP_SCHEDULED_TIME_MINUTE, String.valueOf(config.getBackupTimeMinute()));
    }

    @Override
    @Nonnull
    public Optional<LookAndFeelConfiguration> getLookAndFeelConfiguration() throws PropertyManagerException {
        return getOptionalJsonProperty(LOOK_AND_FEEL_CONFIGURATION_PROPERTY_NAME, LookAndFeelConfiguration.class);
    }

    @Override
    public synchronized void setLookAndFeelConfiguration(LookAndFeelConfiguration lookAndFeelConfiguration, ImageInfo updatedLogoInfo) throws PropertyManagerException {
        final Optional<LookAndFeelConfiguration> oldConfiguration = getLookAndFeelConfiguration();
        setJsonProperty(LOOK_AND_FEEL_CONFIGURATION_PROPERTY_NAME, lookAndFeelConfiguration);
        // Logo is not being removed when updatedLogoInfo is null because using UI there is a possibility to change only
        // some fields, such as "Header". In this case updatedLogoInfo would be null, but previously set logo shouldn't be changed/removed.
        // There is no way to remove only logo from UI so to remove custom logo removeLookAndFeelConfiguration() has to be invoked,
        // which removes whole Look and Feel configuration.
        if (updatedLogoInfo != null) {
            saveBase64Image(LOGO_IMAGE_KEY, updatedLogoInfo);
        }
        eventPublisher.publish(new LookAndFeelUpdatedEvent(oldConfiguration.orElse(null), lookAndFeelConfiguration));
    }

    @Override
    public void removeLookAndFeelConfiguration() throws PropertyManagerException {
        final Optional<LookAndFeelConfiguration> lookAndFeelConfiguration = getLookAndFeelConfiguration();
        removeBase64File(LOGO_IMAGE_KEY);
        removeProperty(LOOK_AND_FEEL_CONFIGURATION_PROPERTY_NAME);
        lookAndFeelConfiguration.ifPresent(config -> eventPublisher.publish(new LookAndFeelUpdatedEvent(config, null)));
    }

    @Override
    public Optional<ImageInfo> getLogoImage() throws PropertyManagerException {
        return getOptionalJsonProperty(String.format(BASE64_FILE_PROPERTY_NAME_FORMAT, PropertyManagerGeneric.LOGO_IMAGE_KEY), ImageInfo.class);
    }

    private void saveBase64Image(String imageKey, ImageInfo imageInfo) throws PropertyManagerException {
        setJsonProperty(String.format(BASE64_FILE_PROPERTY_NAME_FORMAT, imageKey), imageInfo);
    }

    private void removeBase64File(@Nullable String imageKey) {
        if (imageKey != null) {
            removeProperty(String.format(BASE64_FILE_PROPERTY_NAME_FORMAT, imageKey));
        }
    }

    private void setJsonProperty(String property, Object json) throws PropertyManagerException {
        try {
            final String asJson = JSON_MAPPER.writeValueAsString(json);
            setProperty(property, asJson, false);
        } catch (IOException e) {
            throw new PropertyManagerException(e);
        }
    }

    private <T> Optional<T> getOptionalJsonProperty(String property, Class<T> clz) throws PropertyManagerException {
        try {
            final String asJson = getProperty(property);
            return Optional.of(JSON_MAPPER.readValue(asJson, clz));
        } catch (ObjectNotFoundException e) {
            return Optional.empty();
        } catch (IOException e) {
            throw new PropertyManagerException(e);
        }
    }
}
