package com.atlassian.crowd.manager.directory.monitor.poller;

import java.io.Serializable;
import java.util.Map;

import com.atlassian.crowd.directory.DbCachingDirectoryPoller;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.SynchronisableDirectory;
import com.atlassian.crowd.directory.loader.DirectoryInstanceLoader;
import com.atlassian.crowd.directory.monitor.poller.DirectoryPoller;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.manager.directory.DirectorySynchroniser;
import com.atlassian.crowd.manager.directory.SynchronisationMode;
import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.JobRunnerRequest;
import com.atlassian.scheduler.JobRunnerResponse;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.config.JobRunnerKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;

import static com.atlassian.crowd.manager.directory.SynchronisationMode.FULL;
import static com.atlassian.crowd.manager.directory.SynchronisationMode.INCREMENTAL;
import static com.google.common.base.MoreObjects.firstNonNull;

/**
 * @see AtlassianSchedulerDirectoryPollerManager
 */
public class DirectoryPollerJobRunner implements JobRunner {

    private static final Logger log = LoggerFactory.getLogger(DirectoryPollerJobRunner.class);

    public static final JobRunnerKey JOB_RUNNER_KEY = JobRunnerKey.of(DirectoryPollerManager.class.getName());
    public static final String PARAM_DIRECTORY_ID = "DIRECTORY_ID";
    public static final String PARAM_SYNC_MODE = "SYNC_MODE";

    private final DirectoryManager directoryManager;
    private final DirectorySynchroniser directorySynchroniser;
    private final DirectoryInstanceLoader directoryInstanceLoader;
    private final SchedulerService schedulerService;

    public DirectoryPollerJobRunner(DirectoryManager directoryManager,
                                    DirectorySynchroniser directorySynchroniser,
                                    DirectoryInstanceLoader directoryInstanceLoader,
                                    SchedulerService schedulerService) {
        this.directoryManager = directoryManager;
        this.directorySynchroniser = directorySynchroniser;
        this.directoryInstanceLoader = directoryInstanceLoader;
        this.schedulerService = schedulerService;
    }

    public void register() {
        // The JobRunner is registered here instead of by AtlassianSchedulerDirectoryPollerManager
        // to avoid a circular reference between the two
        schedulerService.registerJobRunner(JOB_RUNNER_KEY, this);
    }

    @Override
    public JobRunnerResponse runJob(JobRunnerRequest request) {
        final Map<String, Serializable> params = request.getJobConfig().getParameters();
        final long directoryId = (Long) params.get(PARAM_DIRECTORY_ID);

        final Directory directory;
        try {
            directory = directoryManager.findDirectoryById(directoryId);
        } catch (DirectoryNotFoundException e) {
            log.error("Cannot synchronise unknown directory [ {} ]", directoryId, e);
            return JobRunnerResponse.aborted("Cannot synchronise unknown directory [ " + directoryId + " ]");
        }

        final RemoteDirectory remoteDirectory;
        try {
            remoteDirectory = directoryInstanceLoader.getDirectory(directory);
        } catch (DirectoryInstantiationException e) {
            log.error("Could not instantiate directory {}", directoryId, e);
            return JobRunnerResponse.failed(e);
        }

        if (!(remoteDirectory instanceof SynchronisableDirectory)) {
            log.warn("Directory is not synchronisable. [ {} ]", directoryId);
            return JobRunnerResponse.aborted("Directory is not synchronisable. [ " + directoryId + " ]");
        }

        final SynchronisableDirectory synchronisableDirectory = (SynchronisableDirectory) remoteDirectory;
        final SynchronisationMode synchronisationMode = getSynchronisationMode(params, synchronisableDirectory);
        final DirectoryPoller poller = new DbCachingDirectoryPoller(directorySynchroniser, synchronisableDirectory);
        poller.pollChanges(synchronisationMode);

        return JobRunnerResponse.success();
    }

    private static SynchronisationMode getSynchronisationMode(Map<String, Serializable> params, SynchronisableDirectory directory) {
        return firstNonNull(
                getSynchronisationModeFromJobParams(params), // get sync mode from scheduled job (null for Crowd)
                getSynchronisationModeFromDirectory(directory));
    }

    @Nullable
    private static SynchronisationMode getSynchronisationModeFromJobParams(Map<String, Serializable> params) {
        return (SynchronisationMode) params.get(PARAM_SYNC_MODE);
    }

    private static SynchronisationMode getSynchronisationModeFromDirectory(SynchronisableDirectory directory) {
        return directory.isIncrementalSyncEnabled() ? INCREMENTAL : FULL;
    }

    public void unregister() {
        schedulerService.unregisterJobRunner(JOB_RUNNER_KEY);
    }
}
