package com.atlassian.crowd.manager.directory.monitor;

import com.atlassian.beehive.ClusterLock;
import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.directory.DbCachingDirectoryPoller;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.SynchronisableDirectory;
import com.atlassian.crowd.directory.SynchronisableDirectoryProperties;
import com.atlassian.crowd.directory.loader.DirectoryInstanceLoader;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.spi.DcLicenseChecker;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerJobRunner;
import com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerManager;
import com.atlassian.crowd.service.license.LicenseService;
import com.atlassian.scheduler.JobRunner;
import com.atlassian.scheduler.JobRunnerRequest;
import com.atlassian.scheduler.JobRunnerResponse;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.config.JobConfig;
import com.atlassian.scheduler.config.JobId;
import com.atlassian.scheduler.config.JobRunnerKey;
import com.atlassian.scheduler.config.Schedule;
import com.atlassian.scheduler.status.JobDetails;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.io.Serializable;
import java.time.Clock;
import java.time.Duration;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.atlassian.crowd.directory.SyncScheduleType.CRON_EXPRESSION;
import static com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerJobRunner.PARAM_DIRECTORY_ID;
import static com.atlassian.scheduler.config.RunMode.RUN_ONCE_PER_CLUSTER;
import static java.util.Objects.requireNonNull;

/**
 * Updates scheduled jobs related to directory synchronization, based on the current directory configuration.
 *
 * Synchronisable directories that no longer exist will have their scheduled jobs removed, directories that exist will
 * be created, or updated as needed.
 */
// this mostly exist to separate out interacting with the scheduler from the request threads, and not do that in getDirectory() as we did before
public class DirectoryMonitorRefresherJob implements JobRunner {
    public static final JobRunnerKey JOB_RUNNER_KEY = JobRunnerKey.of(DirectoryMonitorRefresherJob.class.getName() + "-runner");

    static final String LOCK_NAME = DirectoryMonitorRefresherJob.class.getName() + "-lock";
    static final long DEFAULT_POLLING_DELAY = 5000L;
    static final String POLLER_JOBID_PREFIX = DirectoryPollerManager.class.getName() + ".";

    private static final Logger log = LoggerFactory.getLogger(DirectoryMonitorRefresherJob.class);

    private final SchedulerService schedulerService;
    private final DirectoryInstanceLoader directoryInstanceLoader;
    private final DirectoryManager directoryManager;
    private final ClusterLockService clusterLockService;
    private final Clock clock;
    private final DcLicenseChecker licenseChecker;

    public DirectoryMonitorRefresherJob(SchedulerService schedulerService,
                                        DirectoryInstanceLoader directoryInstanceLoader,
                                        DirectoryManager directoryManager,
                                        ClusterLockService clusterLockService,
                                        Clock clock,
                                        DcLicenseChecker licenseChecker) {
        this.schedulerService = schedulerService;
        this.directoryInstanceLoader = directoryInstanceLoader;
        this.directoryManager = directoryManager;
        this.clusterLockService = clusterLockService;
        this.clock = clock;
        this.licenseChecker = licenseChecker;
    }

    @PostConstruct
    public void registerJobRunner() {
        schedulerService.registerJobRunner(DirectoryMonitorRefresherJob.JOB_RUNNER_KEY, this);
    }

    @PreDestroy
    public void unregisterJobRunner() {
        schedulerService.unregisterJobRunner(DirectoryMonitorRefresherJob.JOB_RUNNER_KEY);
    }

    @Override
    public JobRunnerResponse runJob(JobRunnerRequest request) {
        final ClusterLock clusterLock = clusterLockService.getLockForName(LOCK_NAME);

        if (clusterLock.tryLock()) {
            try {
                log.debug("Refreshing directory monitors");

                final Map<JobId, RemoteDirectory> synchronisableDirectories = directoryManager.findAllDirectories().stream()
                        .filter(Directory::isActive)
                        .flatMap(directory -> {
                            try {
                                return Stream.of(directoryInstanceLoader.getDirectory(directory));
                            } catch (DirectoryInstantiationException e) {
                                log.warn("Unable to instantiate directory {} when updating synchronisation schedules", directory.getId());
                                return Stream.empty();
                            }
                        })
                        .filter(directory -> directory instanceof SynchronisableDirectory)
                        .collect(Collectors.toMap(dir -> getJobId(dir.getDirectoryId()), Function.identity()));

                log.debug("Found {} synchronisable directories", synchronisableDirectories.size());

                final List<JobDetails> scheduledJobs = schedulerService
                        .getJobsByJobRunnerKey(DirectoryPollerJobRunner.JOB_RUNNER_KEY)
                        .stream()
                        .filter(jobDetails -> jobDetails.getJobId().toString().startsWith(POLLER_JOBID_PREFIX))
                        .collect(Collectors.toList());

                updateExistingJobs(synchronisableDirectories, scheduledJobs);
                addNewJobs(synchronisableDirectories, scheduledJobs);
            } finally {
                clusterLock.unlock();
            }
        } else {
            log.debug("Lock {} is already held, skipping", LOCK_NAME);
        }

        return JobRunnerResponse.success();
    }

    private void addNewJobs(Map<JobId, RemoteDirectory> expectedJobs, List<JobDetails> scheduledPollingJobs) {
        final Set<JobId> scheduledJobIds = scheduledPollingJobs.stream().map(JobDetails::getJobId).collect(Collectors.toSet());
        expectedJobs.entrySet().stream()
                .filter(job -> !scheduledJobIds.contains(job.getKey()))
                .forEach(missingSyncEntry -> schedulePollingJob(missingSyncEntry.getValue()));
    }

    private void updateExistingJobs(Map<JobId, RemoteDirectory> expectedJobs, List<JobDetails> scheduledPollingJobs) {
        scheduledPollingJobs.forEach(scheduledJob -> {
            final JobId jobId = scheduledJob.getJobId();

            final RemoteDirectory synchronisableDirectory = expectedJobs.get(jobId);
            if (synchronisableDirectory != null) {
                if (shouldReschedule(scheduledJob, synchronisableDirectory)) {
                    log.debug("Synchronisation period differs for directory {} - will reschedule", synchronisableDirectory.getDirectoryId());
                    schedulePollingJob(synchronisableDirectory);
                }
            } else {
                log.debug("Unscheduling polling job {}, as the directory isn't synchronisable", jobId);
                schedulerService.unscheduleJob(jobId);
            }
        });
    }

    private boolean shouldReschedule(JobDetails scheduledJob, RemoteDirectory remoteDirectory) {
        return !normalizeScheduleForComparison(prepareScheduleForDirectory(remoteDirectory))
                .equals(normalizeScheduleForComparison(scheduledJob.getSchedule()));
    }

    private Schedule normalizeScheduleForComparison(Schedule schedule) {
        if (schedule.getIntervalScheduleInfo() != null) {
            return Schedule.forInterval(schedule.getIntervalScheduleInfo().getIntervalInMillis(), null);
        }
        return schedule;
    }

    private void schedulePollingJob(RemoteDirectory directory) {
        final JobConfig jobConfig = createPollingJobConfig(directory);
        final JobId jobId = getJobId(directory.getDirectoryId());

        log.debug("Scheduling polling job {}, with schedule {}", jobId, jobConfig.getSchedule());
        try {
            schedulerService.scheduleJob(jobId, jobConfig);
        } catch (SchedulerServiceException e) {
            log.error("Failed to schedule directory polling job {}", jobId);
        }
    }

    private JobId getJobId(long directoryId) {
        return JobId.of(POLLER_JOBID_PREFIX + directoryId);
    }

    private JobConfig createPollingJobConfig(RemoteDirectory directory) {
        long directoryId = directory.getDirectoryId();

        return JobConfig.forJobRunnerKey(DirectoryPollerJobRunner.JOB_RUNNER_KEY)
                .withRunMode(RUN_ONCE_PER_CLUSTER)
                .withSchedule(prepareScheduleForDirectory(directory))
                .withParameters(ImmutableMap.<String, Serializable>builder()
                        .put(PARAM_DIRECTORY_ID, directoryId)
                        .build());
    }

    private Schedule prepareScheduleForDirectory(RemoteDirectory remoteDirectory) {
        if (Schedule.Type.INTERVAL.equals(getScheduleType(remoteDirectory)) || !licenseChecker.isDcLicense()) {
            return Schedule.forInterval(getInterval(remoteDirectory).toMillis(), new Date(getFirstRunTime()));
        }

        String value = requireNonNull(remoteDirectory.getValue(SynchronisableDirectoryProperties.CACHE_SYNCHRONISE_CRON));
        return Schedule.forCronExpression(value);
    }

    private Duration getInterval(RemoteDirectory remoteDirectory) {
        return DbCachingDirectoryPoller.getPollingInterval(remoteDirectory);
    }

    private long getFirstRunTime() {
        return clock.millis() + Long.getLong("crowd.polling.startdelay", DEFAULT_POLLING_DELAY);
    }

    private Schedule.Type getScheduleType(RemoteDirectory remoteDirectory) {
        String synchronizationOption = remoteDirectory.getValue(SynchronisableDirectoryProperties.CACHE_SYNCHRONISATION_TYPE);
        if (CRON_EXPRESSION.equals(synchronizationOption)) {
            return Schedule.Type.CRON_EXPRESSION;
        }

        return Schedule.Type.INTERVAL;
    }
}