package com.atlassian.crowd.manager.directory;

import com.atlassian.crowd.directory.DirectorySynchronisationStatusImpl;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationRoundInformation;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.embedded.spi.DirectorySynchronisationStatusDao;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.model.directory.DirectorySynchronisationStatus;
import com.atlassian.crowd.model.directory.SynchronisationStatusKey;
import com.atlassian.crowd.service.cluster.ClusterNode;
import com.atlassian.crowd.service.cluster.ClusterService;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static com.atlassian.crowd.mapper.DirectorySynchronisationStatusMapper.mapDirectoryStatusToRoundInformation;

/**
 * Stores information about synchronisation status in the database
 *
 * @since 2.12.0
 */
@Transactional
public class InDatabaseDirectorySynchronisationInformationStore implements DirectorySynchronisationInformationStore {
    private final static Logger logger = LoggerFactory.getLogger(InDatabaseDirectorySynchronisationInformationStore.class);

    private final DirectorySynchronisationStatusDao statusDao;
    private final DirectoryDao directoryDao;
    private final ClusterService clusterService;

    public InDatabaseDirectorySynchronisationInformationStore(DirectorySynchronisationStatusDao statusDao, DirectoryDao directoryDao, ClusterService clusterService) {
        this.statusDao = statusDao;
        this.directoryDao = directoryDao;
        this.clusterService = clusterService;
    }

    @Override
    public DirectorySynchronisationRoundInformation getActive(long directoryId) {
        final Optional<DirectorySynchronisationStatus> statusForActiveRound = statusDao.findActiveForDirectory(directoryId);
        if (statusForActiveRound.isPresent()) {
            final DirectorySynchronisationStatus st = statusForActiveRound.get();
            return mapDirectoryStatusToRoundInformation(st);
        }
        return null;
    }

    @Override
    public Optional<DirectorySynchronisationRoundInformation> getLast(long directoryId) {
        final Optional<DirectorySynchronisationStatus> statusForLastRound = statusDao.findLastForDirectory(directoryId);
        if (statusForLastRound.isPresent()) {
            final DirectorySynchronisationStatus st = statusForLastRound.get();
            logger.debug("Successfully restored last synchronisation status for directory {}", directoryId);
            return Optional.of(mapDirectoryStatusToRoundInformation(st));
        } else {
            logger.debug("Didn't find status of last synchronisation for directory {}, that's normal for the very first synchronisation", directoryId);
            return Optional.empty();
        }
    }

    @Override
    public void clear(long directoryId) {
        statusDao.removeStatusesForDirectory(directoryId);
        logger.debug("Cleared synchronisation statuses for directory {}", directoryId);
    }

    @Override
    public void clear() {
        statusDao.removeAll();
        logger.debug("Removed all synchronisation statuses");
    }

    @Override
    public void syncStatus(long directoryId, String statusKey, List<Serializable> parameters) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void syncStatus(long directoryId, SynchronisationStatusKey statusKey, List<Serializable> parameters) {
        final Optional<DirectorySynchronisationStatus> previous = statusDao.findActiveForDirectory(directoryId);
        try {
            if (previous.isPresent()) {
                statusDao.update(getStatusBuilder(previous.get())
                        .setStatus(statusKey, parameters)
                        .build());
            } else {
                logger.info("Got synchronisation status update for directory {} with status {}, but didn't find any active status in the database, creating new record instead", directoryId, statusKey);
                statusDao.add(getStatusBuilder()
                        .setDirectory(directoryDao.findById(directoryId))
                        .setStartTimestamp(System.currentTimeMillis())
                        .setStatus(statusKey, parameters)
                        .build());
            }
        } catch (ObjectNotFoundException | DirectoryNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void syncStarted(long directoryId, long timestamp) {
        try {
            final Optional<DirectorySynchronisationStatus> active = statusDao.findActiveForDirectory(directoryId);
            if (active.isPresent()) {
                logger.warn("Found active synchronisation status during start of new synchronisation. This may indicate that the previous synchronisation didn't end up correctly");
                statusDao.update(getStatusBuilder(active.get())
                        .setStartTimestamp(timestamp)
                        .setEndTimestamp(null)
                        .setStatus(SynchronisationStatusKey.STARTED, Collections.emptyList())
                        .build());
            } else {
                statusDao.add(getStatusBuilder()
                        .setDirectory(directoryDao.findById(directoryId))
                        .setStartTimestamp(timestamp)
                        .setStatus(SynchronisationStatusKey.STARTED, Collections.emptyList())
                        .build());
            }
        } catch (ObjectNotFoundException | DirectoryNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void syncFailure(long directoryId, SynchronisationMode syncMode, String failureReason) {
        final Optional<DirectorySynchronisationStatus> previous = statusDao.findActiveForDirectory(directoryId);
        try {
            if (previous.isPresent()) {
                statusDao.update(getStatusBuilder(previous.get())
                        .setSyncError(syncMode, failureReason)
                        .build());
            } else {
                logger.info("Got synchronisation failure for directory {}, but didn't find any active status in the database, creating new record instead", directoryId);
                statusDao.add(getStatusBuilder()
                        .setDirectory(directoryDao.findById(directoryId))
                        .setStartTimestamp(System.currentTimeMillis())
                        .setStatus(SynchronisationStatusKey.STARTED, Collections.emptyList())
                        .setSyncError(syncMode, failureReason)
                        .build());
            }
        } catch (ObjectNotFoundException | DirectoryNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void syncFinished(long directoryId, long timestamp, SynchronisationStatusKey statusKey, List<Serializable> parameters) {
        final Optional<DirectorySynchronisationStatus> active = statusDao.findActiveForDirectory(directoryId);

        try {
            if (active.isPresent()) {
                final DirectorySynchronisationStatus status = active.get();
                statusDao.removeAllExcept(directoryId, status.getId());
                statusDao.update(getStatusBuilder(active.get())
                    .setEndTimestamp(timestamp)
                    .setStatus(statusKey, parameters)
                    .build());
            } else {
                logger.warn("Didn't find active synchronisation status during finish of the synchronisation");
                statusDao.removeStatusesForDirectory(directoryId);
                statusDao.add(getStatusBuilder()
                        .setDirectory(directoryDao.findById(directoryId))
                        .setStartTimestamp(timestamp)
                        .setEndTimestamp(timestamp)
                        .setStatus(statusKey, parameters)
                        .build());
            }
        } catch (ObjectNotFoundException | DirectoryNotFoundException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public Collection<DirectorySynchronisationStatus> getStalledSynchronizations() {
        if (!clusterService.isAvailable()) {
            logger.debug("Ran InDatabaseDirectorySynchronisationInformationStore#getStalledSynchronizations in non-cluster configuration");
            return ImmutableList.of();
        }
        final Set<String> activeNodesIds = clusterService.getInformation().getNodes().stream().map(ClusterNode::getNodeId).collect(Collectors.toSet());
        if (activeNodesIds.isEmpty()) {
            logger.warn("Crowd is running in cluster configuration but wasn't able to find any active nodes");
            return ImmutableList.of();
        }
        return statusDao.findActiveSyncsWhereNodeIdNotIn(activeNodesIds);
    }

    private DirectorySynchronisationStatusImpl.Builder getStatusBuilder() {
        return updateNodeInfo(
                DirectorySynchronisationStatusImpl.builder());
    }

    private DirectorySynchronisationStatusImpl.Builder getStatusBuilder(DirectorySynchronisationStatus previousStatus) {
        return updateNodeInfo(
                DirectorySynchronisationStatusImpl.builder(previousStatus));
    }

    private DirectorySynchronisationStatusImpl.Builder updateNodeInfo(DirectorySynchronisationStatusImpl.Builder builder) {
        Optional<ClusterNode> clusterNode = clusterService.getClusterNode();
        return builder
                .setNodeId(clusterNode.map(ClusterNode::getNodeId).orElse(null))
                .setNodeName(clusterNode.map(ClusterNode::getNodeName).orElse(null));
    }
}
