package com.atlassian.crowd.manager.directory;

import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.audit.AuditLogContext;
import com.atlassian.crowd.audit.AuditLogEventSource;
import com.atlassian.crowd.directory.SynchronisableDirectory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationRoundInformation;
import com.atlassian.crowd.event.directory.RemoteDirectorySynchronisationStartedEvent;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.directory.SynchronisationStatusKey;
import com.atlassian.crowd.util.DirectorySynchronisationEventHelper;
import com.atlassian.event.api.EventPublisher;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.concurrent.locks.Lock;

/**
 * An implementation of a {@link DirectorySynchroniser}.
 */
public class DirectorySynchroniserImpl implements DirectorySynchroniser {
    private final static Logger log = LoggerFactory.getLogger(DirectorySynchroniser.class);

    private final ClusterLockService lockService;
    private final DirectorySynchroniserHelper directorySynchroniserHelper;
    private final InternalSynchronisationStatusManager synchronisationStatusManager;
    private final EventPublisher eventPublisher;
    private final AuditLogContext auditLogContext;
    private final DirectorySynchronisationEventHelper syncEventHelper;

    public DirectorySynchroniserImpl(ClusterLockService lockService,
                                     DirectorySynchroniserHelper directorySynchroniserHelper,
                                     InternalSynchronisationStatusManager synchronisationStatusManager,
                                     EventPublisher eventPublisher,
                                     AuditLogContext auditLogContext,
                                     DirectorySynchronisationEventHelper syncEventHelper) {
        this.lockService = lockService;
        this.directorySynchroniserHelper = directorySynchroniserHelper;
        this.synchronisationStatusManager = synchronisationStatusManager;
        this.eventPublisher = eventPublisher;
        this.auditLogContext = auditLogContext;
        this.syncEventHelper = syncEventHelper;
    }

    /**
     * Synchronises a remote directory. For performance reasons, this method must not run within
     * a transaction. As a synchronisation may involve a huge amount of data, individual batches
     * are run in their own transactions. Running overall synchronisation in a transaction would defeat
     * this and is prevented.
     */
    @Transactional(propagation = Propagation.NEVER)
    @Override
    public void synchronise(final SynchronisableDirectory remoteDirectory, final SynchronisationMode mode)
            throws DirectoryNotFoundException, OperationFailedException {
        long directoryId = remoteDirectory.getDirectoryId();
        final Directory directory = findDirectoryById(directoryId);
        if (!directory.isActive()) {
            log.debug("Request to synchronise directory [ {} ] in {} mode is returning silently because the directory is not active.",
                    directoryId, mode);
            return;
        }

        log.debug("request to synchronise directory [ {} ] in {} mode", directoryId, mode);

        final Lock lock = lockService.getLockForName(DirectorySynchronisationUtils.getLockName(directoryId));

        if (lock.tryLock()) {
            boolean successful = false;
            try {
                directorySynchroniserHelper.updateSyncStartTime(remoteDirectory);
                synchronisationStatusManager.syncStarted(directory);
                try {
                    auditLogContext.withAuditLogSource(AuditLogEventSource.SYNCHRONIZATION, () -> {
                        eventPublisher.publish(new RemoteDirectorySynchronisationStartedEvent(remoteDirectory));
                        remoteDirectory.synchroniseCache(mode, synchronisationStatusManager);
                        return null;
                    });
                    finishSynchronisationIfWasNotFinishedAlready(directory);
                    successful = true;
                } catch (Exception e) {
                    finishSynchronisationAsFailedIfWasNotFinishedAlready(directory, e);
                    Throwables.propagateIfPossible(e, DirectoryNotFoundException.class, OperationFailedException.class);
                } finally {
                    directorySynchroniserHelper.updateSyncEndTime(remoteDirectory);
                    publishSynchronisationEndedEvent(directory, successful);
                }
            } finally {
                lock.unlock();

            }
        } else {
            log.debug("directory [ {} ] already synchronising", directoryId);
        }
    }

    @VisibleForTesting
    void finishSynchronisationAsFailedIfWasNotFinishedAlready(Directory directory, Exception e) {
        final DirectorySynchronisationRoundInformation activeRound = synchronisationStatusManager.getDirectorySynchronisationInformation(directory).getActiveRound();
        if (activeRound != null) {
            synchronisationStatusManager.syncFailure(directory.getId(), SynchronisationMode.FULL, e);
            synchronisationStatusManager.syncFinished(directory.getId(),
                    resolveKey(activeRound, false), ImmutableList.of());
        }
    }

    @VisibleForTesting
    void finishSynchronisationIfWasNotFinishedAlready(Directory directory) {
        final DirectorySynchronisationRoundInformation activeRound = synchronisationStatusManager.getDirectorySynchronisationInformation(directory).getActiveRound();
        if (activeRound != null) {
            synchronisationStatusManager.syncFinished(directory.getId(),
                    resolveKey(activeRound, true), ImmutableList.of());
        }
    }

    private SynchronisationStatusKey resolveKey(DirectorySynchronisationRoundInformation activeRound, boolean successful) {
        // the synchronisation might've been incremental, however `SynchronisableDirectory` does not provide
        // any information about the actual mode. This status will only be set if either the status write failed
        // or the `SynchronisableDirectory` implementation did not mark the round as completed.
        return SynchronisationStatusKey.fromKey(activeRound.getStatusKey())
                .filter(SynchronisationStatusKey::isFinal)
                .orElse(successful ? SynchronisationStatusKey.SUCCESS_FULL : SynchronisationStatusKey.FAILURE);
    }

    private void publishSynchronisationEndedEvent(final Directory directory, final boolean wasSuccessful) {
        try {
            syncEventHelper.publishDirectorySynchronisationEvent(this, directory, wasSuccessful, null);
        } catch (Exception e) {
            throw new RuntimeException("Could not publish synchronisation ended event", e);
        }
    }

    public boolean isSynchronising(long directoryId) throws DirectoryNotFoundException {
        return directorySynchroniserHelper.isSynchronising(directoryId);
    }

    private Directory findDirectoryById(final long directoryId) throws DirectoryNotFoundException {
        return directorySynchroniserHelper.findDirectoryById(directoryId);
    }
}
