package com.atlassian.crowd.manager.directory;

import com.atlassian.crowd.embedded.api.DirectorySynchronisationRoundInformation;
import com.atlassian.crowd.model.directory.DirectorySynchronisationStatus;
import com.atlassian.crowd.model.directory.SynchronisationStatusKey;

import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

/**
 * Storage for directory synchronisation information.
 */
public interface DirectorySynchronisationInformationStore {
    /**
     * Provides information about the status of the active synchronisation for the directory.
     *
     * @param directoryId directory id
     * @return information about the active synchronisation if present
     */
    @Nullable
    DirectorySynchronisationRoundInformation getActive(long directoryId);


    /**
     * Provides information about the last completed (either succeeded or failed) synchronisation.
     *
     * @param directoryId directory id
     * @return an optional either containing information about the last synchronisation if present
     */
    Optional<DirectorySynchronisationRoundInformation> getLast(long directoryId);

    /**
     * Clears directory synchronisation information for the given directory.
     *
     * @param directoryId directory id
     */
    void clear(long directoryId);

    /**
     * Clears directory synchronisation information of all directories.
     */
    void clear();

    /**
     * Updates active directory synchronisation status
     *
     * @param directoryId directory id
     * @param statusKey   new status i18n key
     * @param parameters  parameters for the statusKey
     */
    void syncStatus(long directoryId, String statusKey, List<Serializable> parameters);

    /**
     * Updates active directory synchronisation status.
     *
     * @param directoryId directory id
     * @param statusKey   new status i18n key
     * @param parameters  paramaters for the statusKey
     */
    void syncStatus(long directoryId, SynchronisationStatusKey statusKey, List<Serializable> parameters);

    /**
     * Starts the synchronisation, creating new directory synchronisation information.
     *
     * @param directoryId directory id
     * @param timestamp   synchronisation start
     */
    void syncStarted(long directoryId, long timestamp);

    /**
     * Updates active directory synchronisation status with a failure reason.
     *
     * @param directoryId   directory id
     * @param syncMode      synchronisation mode
     * @param failureReason failure reason
     */
    default void syncFailure(long directoryId, SynchronisationMode syncMode, String failureReason) {
        throw new UnsupportedOperationException();
    }

    /**
     * Finishes synchronisation status, marking it as last status information and clearing active synchronisation status.
     *
     * @param directoryId directory id
     * @param timestamp   synchronisation end
     * @param statusKey   ending status (should indicate failure or success)
     * @param parameters  parameters for the statusKey
     */
    void syncFinished(long directoryId, long timestamp, SynchronisationStatusKey statusKey, List<Serializable> parameters);

    /**
     * Attempts to find synchronizations that have been marked as in-progress, but don't seem to be running anymore.
     * The exact method of finding those depends on the implementation and configuration.
     * @return a list of synchronizations that are potentially stalled, or an empty list if stalled synchronisations can't be determined.
     */
    Collection<DirectorySynchronisationStatus> getStalledSynchronizations();
}
