package com.atlassian.crowd.manager.application.search;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.Query;
import com.atlassian.crowd.embedded.impl.IdentifierMap;
import com.atlassian.crowd.manager.application.canonicality.CanonicalityChecker;
import com.atlassian.crowd.model.NameComparator;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.query.QueryUtils;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.search.util.ResultsAggregator;
import com.atlassian.crowd.search.util.ResultsAggregators;
import com.atlassian.crowd.search.util.SearchResultsUtil;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimaps;
import org.apache.commons.lang3.tuple.Pair;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Helper class for running and aggregating results of the query across multiple directories.
 */
class InMemoryQueryRunner<T, Q extends Query<T>> {
    private static final int MIN_OPTIMISTIC_QUERY_MARGIN = 10;

    private final DirectoryManagerSearchWrapper directoryManagerSearchWrapper;
    private final List<Directory> directories;
    private final EntityDescriptor entityDescriptor;
    private final CanonicalityChecker canonicalityChecker;
    private final boolean mergeEntities;
    private final boolean nested;
    private final Q query;
    private final Q splitQuery;
    // All directories, except first, have unknown number of non-canonical results filtered out. Instead of
    // querying for all results, run this optimistic query first and re-run if needed.
    private final Q optimisticSplitQuery;
    private final Q withAllResults;
    private final Directory firstDir;
    private final BiFunction<Directory, Q, Optional<DirectoryQueryWithFilter<T>>> queryProvider;


    private InMemoryQueryRunner(DirectoryManagerSearchWrapper directoryManagerSearchWrapper,
                        List<Directory> directories,
                        CanonicalityChecker canonicalityChecker, // null if check not needed
                        boolean mergeEntities,
                        boolean nested,
                        BiFunction<Directory, Q, Optional<DirectoryQueryWithFilter<T>>> queryProvider,
                        EntityDescriptor entityDescriptor,
                        Q query,
                        Q splitQuery,
                        Q optimisticSplitQuery,
                        Q withAllResults) {
        this.directoryManagerSearchWrapper = directoryManagerSearchWrapper;
        this.directories = directories;
        this.canonicalityChecker = canonicalityChecker;
        this.mergeEntities = mergeEntities;
        this.nested = nested;
        this.firstDir= directories.get(0);
        this.queryProvider = queryProvider;
        this.entityDescriptor = entityDescriptor;
        this.query = query;
        this.splitQuery = splitQuery;
        this.optimisticSplitQuery = optimisticSplitQuery;
        this.withAllResults = withAllResults;
    }

    public List<T> search() {
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class, User.class);
        if (canonicalityChecker == null) {
            List<DirectoryQueryWithFilter<T>> queries = prepareValidQueries(true);
            return merge(this::execute, queries);
        } else {
            return searchWithCanonicalityCheck();
        }
    }

    private List<T> searchWithCanonicalityCheck() {
        final ListMultimap<Long, T> resultsByDirectory = LinkedListMultimap.create();
        List<Pair<DirectoryQueryWithFilter<T>, T>> rerunCandidates = new ArrayList<>();
        List<DirectoryQueryWithFilter<T>> queries = prepareValidQueries(true);
        for (DirectoryQueryWithFilter<T> validQuery : queries) {
            List<T> dirResults = execute(validQuery, rerunCandidates);
            resultsByDirectory.putAll(validQuery.getDirectory().getId(), dirResults);
        }
        final List<T> result = removeNonCanonicalEntitiesIfNeeded(resultsByDirectory, queries);
        final boolean rerun = rerunIfNeeded(rerunCandidates, resultsByDirectory, result);
        return rerun ? removeNonCanonicalEntitiesIfNeeded(resultsByDirectory, queries) : result;
    }

    private List<T> execute(DirectoryQueryWithFilter<T> validQuery) {
        return validQuery.filterResults(executeUnfiltered(validQuery));
    }

    private List<T> execute(DirectoryQueryWithFilter<T> validQuery, List<Pair<DirectoryQueryWithFilter<T>, T>> rerunCandidates) {
        final List<T> unfiltered = executeUnfiltered(validQuery);
        if (!validQuery.getDirectory().getId().equals(firstDir.getId())
                && query.getMaxResults() != EntityQuery.ALL_RESULTS
                && validQuery.getQuery().getMaxResults() != EntityQuery.ALL_RESULTS
                && unfiltered.size() >= validQuery.getQuery().getMaxResults()) {
            // Only if there are more results to fetch it makes sense to re-query. Also keep the last result,
            // because if it's too high there is no point in asking for more.
            rerunCandidates.add(Pair.of(validQuery, unfiltered.get(unfiltered.size() - 1)));
        }
        return validQuery.filterResults(unfiltered);
    }

    private boolean rerunIfNeeded(List<Pair<DirectoryQueryWithFilter<T>, T>> rerunCandidates,
                                  ListMultimap<Long, T> resultsByDirectory,
                                  List<T> result) {
        if (rerunCandidates.isEmpty()) {
            return false;
        }
        final T last = result.size() < query.getMaxResults() ? null : result.get(result.size() - 1);
        final Comparator<T> cmp = NameComparator.of(query.getReturnType());
        boolean rerun = false;
        for (Pair<DirectoryQueryWithFilter<T>, T> candidate : rerunCandidates) {
            Long directoryId = candidate.getLeft().getDirectory().getId();
            if (resultsByDirectory.get(directoryId).size() < splitQuery.getMaxResults()
                    && (last == null || cmp.compare(candidate.getRight(), last) < 0)) {
                resultsByDirectory.replaceValues(directoryId, execute(candidate.getLeft().getDirectory(), withAllResults));
                rerun = true;
            }
        }
        return rerun;
    }

    private List<T> execute(Directory directory, Q query) {
        return queryProvider.apply(directory, query)
                .map(dirQuery -> dirQuery.filterResults(executeUnfiltered(dirQuery)))
                .orElse(ImmutableList.of());
    }

    private List<T> executeUnfiltered(DirectoryQueryWithFilter<T> directoryQueryWithFilter) {
        Long directoryId = directoryQueryWithFilter.getDirectory().getId();
        if (directoryQueryWithFilter.getQuery() instanceof MembershipQuery) {
            return nested ? directoryManagerSearchWrapper.searchNestedGroupRelationships(directoryId, directoryQueryWithFilter.getMembershipQuery())
                    : directoryManagerSearchWrapper.searchDirectGroupRelationships(directoryId, directoryQueryWithFilter.getMembershipQuery());
        } else {
            return directoryManagerSearchWrapper.search(directoryId, (EntityQuery<T>) directoryQueryWithFilter.getQuery());
        }
    }

    private List<T> removeNonCanonicalEntitiesIfNeeded(final ListMultimap<Long, T> results, List<DirectoryQueryWithFilter<T>> queries) {
        if (canonicalityChecker != null) {
            canonicalityChecker.removeNonCanonicalEntities(
                    Multimaps.transformValues(results, NameComparator.nameGetter(query.getReturnType())),
                    entityDescriptor);
        }
        return merge(helper -> results.get(helper.getDirectory().getId()), queries);
    }

    private List<T> merge(Function<DirectoryQueryWithFilter<T>, List<T>> provider, List<DirectoryQueryWithFilter<T>> queries) {
        if (directories.size() == 1 && queries.size() == 1) {
            Query<T> effectiveQuery = queries.get(0).getQuery();
            if (effectiveQuery.getStartIndex() == query.getStartIndex()) {
                return SearchResultsUtil.constrainResults(provider.apply(queries.get(0)), 0, query.getMaxResults());
            }
        }
        ResultsAggregator<T> aggregator = ResultsAggregators.with(query, mergeEntities);
        queries.forEach(directoryQuery -> aggregator.addAll(provider.apply(directoryQuery)));
        return aggregator.constrainResults();
    }

    public ListMultimap<String, T> searchGroupedByName() {
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class, User.class);

        MembershipQuery<T> membershipQuery = (MembershipQuery<T>) query;

        final IdentifierMap<ListMultimap<Long, T>> perGroup = new IdentifierMap<>(Maps.toMap(
                membershipQuery.getEntityNamesToMatch(), name -> LinkedListMultimap.create()));
        List<DirectoryQueryWithFilter<T>> queries = prepareValidQueries(false);
        for (DirectoryQueryWithFilter<T> validQuery : queries) {
            Long directoryId = validQuery.getDirectory().getId();
            final ListMultimap<String, T> unfiltered = directoryManagerSearchWrapper.searchDirectGroupRelationshipsGroupedByName(
                    directoryId, validQuery.getMembershipQuery());
            for (String name : unfiltered.keySet()) {
                List<T> filtered = validQuery.filterResults(unfiltered.get(name));
                perGroup.get(name).putAll(directoryId, filtered);
            }
        }

        final ListMultimap<String, T> result = ArrayListMultimap.create();
        for (String name : membershipQuery.getEntityNamesToMatch()) {
            result.putAll(name, removeNonCanonicalEntitiesIfNeeded(perGroup.get(name), queries));
        }
        return result;
    }

    private List<DirectoryQueryWithFilter<T>> prepareValidQueries(boolean allowReRun) {
        final Q furtherDirectories;
        if (canonicalityChecker == null) {
            furtherDirectories = splitQuery;
        } else {
            furtherDirectories = (allowReRun && !nested) ? optimisticSplitQuery : withAllResults;
        }
        final List<DirectoryQueryWithFilter<T>> queries = new ArrayList<>(directories.size());
        queryProvider.apply(firstDir, directories.size() == 1 ? query : splitQuery).ifPresent(queries::add);
        for (Directory directory : directories.subList(1, directories.size())) {
            queryProvider.apply(directory, furtherDirectories).ifPresent(queries::add);
        }
        return queries.stream().filter(this::isValid).collect(Collectors.toList());
    }

    private boolean isValid(DirectoryQueryWithFilter<T> directoryQueryWithFilter) {
        if (directoryQueryWithFilter.getQuery() instanceof MembershipQuery) {
            return !directoryQueryWithFilter.getMembershipQuery().getEntityNamesToMatch().isEmpty();
        }
        return true;
    }

    static <T> InMemoryQueryRunner<T, EntityQuery<T>> createEntityQueryRunner(
            DirectoryManagerSearchWrapper directoryManagerSearchWrapper,
            List<Directory> directories,
            CanonicalityChecker canonicalityChecker, // null if check not needed
            boolean mergeEntities,
            BiFunction<Directory, EntityQuery<T>, Optional<DirectoryQueryWithFilter<T>>> queryProvider,
            EntityQuery<T> query) {
        EntityQuery<T> splitQuery = query.baseSplitQuery();
        EntityQuery<T> optimisticSplitQuery = splitQuery.addToMaxResults(Math.max(splitQuery.getMaxResults(), MIN_OPTIMISTIC_QUERY_MARGIN));
        return new InMemoryQueryRunner<>(
            directoryManagerSearchWrapper, directories, canonicalityChecker, mergeEntities, false, queryProvider,
                query.getEntityDescriptor(), query, splitQuery, optimisticSplitQuery, query.withAllResults());
    }

    static <T> InMemoryQueryRunner<T, MembershipQuery<T>> createMembershipQueryRunner(
            DirectoryManagerSearchWrapper directoryManagerSearchWrapper,
            List<Directory> directories,
            CanonicalityChecker canonicalityChecker, // null if check not needed
            boolean mergeEntities,
            boolean nested,
            BiFunction<Directory, MembershipQuery<T>, Optional<DirectoryQueryWithFilter<T>>> queryProvider,
            MembershipQuery<T> query) {
        MembershipQuery<T> splitQuery = query.baseSplitQuery();
        MembershipQuery<T> optimisticSplitQuery = splitQuery.addToMaxResults(Math.max(splitQuery.getMaxResults(), MIN_OPTIMISTIC_QUERY_MARGIN));
        return new InMemoryQueryRunner<>(
                directoryManagerSearchWrapper, directories, canonicalityChecker, mergeEntities, nested, queryProvider,
                query.getEntityToReturn(), query, splitQuery, optimisticSplitQuery, query.withAllResults());
    }
}
