package com.atlassian.crowd.manager.application.search;

import com.atlassian.crowd.common.properties.SystemProperties;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.ListMultimap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ConcurrentModificationException;
import java.util.List;

/**
 * Helper class for {@link DirectoryManager} search methods.
 */
public class DirectoryManagerSearchWrapper {
    private static final Logger log = LoggerFactory.getLogger(DirectoryManagerSearchWrapper.class);

    final DirectoryManager directoryManager;

    public DirectoryManagerSearchWrapper(DirectoryManager directoryManager) {
        this.directoryManager = directoryManager;
    }

    public <T> List<T> search(long directoryId, EntityQuery<T> query) {
        final Entity entityType = query.getEntityDescriptor().getEntityType();
        switch (entityType) {
            case USER:
                return searchUsers(directoryId, query);
            case GROUP:
                return searchGroups(directoryId, query);
            default:
                throw new IllegalArgumentException("Can't query for " + entityType);
        }
    }

    public <T> List<T> searchUsers(long directoryId, EntityQuery<T> query) {
        return handle(() -> directoryManager.searchUsers(directoryId, query), ImmutableList.of());
    }

    public <T> List<T> searchGroups(long directoryId, EntityQuery<T> query) {
        return handle(() -> directoryManager.searchGroups(directoryId, query), ImmutableList.of());
    }

    public <T> List<T> searchDirectGroupRelationships(long directoryId, MembershipQuery<T> query) {
        return handle(() -> directoryManager.searchDirectGroupRelationships(directoryId, query), ImmutableList.of());
    }

    public <T> List<T> searchNestedGroupRelationships(long directoryId, MembershipQuery<T> query) {
        return handle(() -> directoryManager.searchNestedGroupRelationships(directoryId, query), ImmutableList.of());
    }

    public <T> ListMultimap<String, T> searchDirectGroupRelationshipsGroupedByName(long directoryId, MembershipQuery<T> query) {
        return handle(() -> directoryManager.searchDirectGroupRelationshipsGroupedByName(directoryId, query), ImmutableListMultimap.of());
    }

    private <T> T handle(final Operation<T> operation, T defaultValue) {
        try {
            return operation.execute();
        } catch (DirectoryNotFoundException e) {
            throw new ConcurrentModificationException("Directory mapping was removed while iterating through directories", e);
        } catch (OperationFailedException e) {
            if (SystemProperties.SWALLOW_EXCEPTIONS_IN_DIRECTORY_SEARCH.getValue()) {
                log.error("Failed to search underlying directory", e);
            } else {
                throw new com.atlassian.crowd.exception.runtime.OperationFailedException(e);
            }
        }
        return defaultValue;
    }

    private interface Operation<T> {
        T execute() throws OperationFailedException, DirectoryNotFoundException;
    }
}
