package com.atlassian.crowd.manager.application.filtering;

import com.atlassian.crowd.embedded.impl.IdentifierMap;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.manager.application.search.DirectoryManagerSearchWrapper;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Helper class for filtering groups with access. Fetches groups with access lazily.
 */
class GroupFilter {
    private DirectoryManagerSearchWrapper searcher;
    private final long directoryId;
    private final IdentifierMap<String> withAccess;
    private Set<String> toCheck;

    GroupFilter(DirectoryManagerSearchWrapper searcher, ApplicationDirectoryMapping mapping) {
        Preconditions.checkArgument(!mapping.isAllowAllToAuthenticate());
        this.searcher = searcher;
        this.directoryId = mapping.getDirectory().getId();
        this.withAccess = new IdentifierMap<>();
        mapping.getAuthorisedGroupNames().forEach(e -> withAccess.put(e, e));
        this.toCheck = new HashSet<>(mapping.getAuthorisedGroupNames());
    }

    boolean hasAccess(final String name) {
        return anyHasAccess(ImmutableList.of(name));
    }

    boolean anyHasAccess(final Collection<String> names) {
        Collection<String> lowerCasedNames = IdentifierUtils.toLowerCase(names);
        if (anyHasAccessInternal(lowerCasedNames)) {
            return true;
        }
        while (!toCheck.isEmpty()) {
            fetchNext();
            if (anyHasAccessInternal(lowerCasedNames)) {
                return true;
            }
        }
        return false;
    }

    private boolean anyHasAccessInternal(Collection<String> lowerCasedNames) {
        for (String lowerCasedName : lowerCasedNames) {
            if (withAccess.keySet().contains(lowerCasedName)) {
                return true;
            }
        }
        return false;
    }

    Collection<String> getAllWithAccess() {
        while (!toCheck.isEmpty()) {
            fetchNext();
        }
        return withAccess.values();
    }

    private void fetchNext() {
        MembershipQuery<String> membershipQuery = QueryBuilder.queryFor(String.class, EntityDescriptor.group())
                .childrenOf(EntityDescriptor.group())
                .withNames(toCheck)
                .returningAtMost(EntityQuery.ALL_RESULTS);
        List<String> names = searcher.searchDirectGroupRelationships(directoryId, membershipQuery);
        toCheck = names.stream().filter(e -> withAccess.put(e, e) == null).collect(Collectors.toSet());
    }

    public boolean isEmpty() {
        return withAccess.isEmpty();
    }
}
