package com.atlassian.crowd.manager.application;

import com.atlassian.crowd.common.util.ProxyUtil;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.manager.application.canonicality.CanonicalityChecker;
import com.atlassian.crowd.manager.application.canonicality.OptimizedCanonicalityChecker;
import com.atlassian.crowd.manager.application.filtering.AccessFilter;
import com.atlassian.crowd.manager.application.filtering.AccessFilterFactory;
import com.atlassian.crowd.manager.application.search.MembershipSearchStrategy;
import com.atlassian.crowd.manager.application.search.SearchStrategyFactory;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.group.ImmutableMembership;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import static com.atlassian.crowd.model.application.Applications.getActiveDirectories;

public class MembershipsIterableImpl implements ApplicationService.MembershipsIterable {
    private static final Logger logger = LoggerFactory.getLogger(MembershipsIterableImpl.class);
    private static final int GET_MEMBERSHIPS_BATCH_SIZE = 1000;
    private static final EntityQuery<String> ALL_GROUPS_QUERY =
            QueryBuilder.queryFor(String.class, EntityDescriptor.group()).returningAtMost(EntityQuery.ALL_RESULTS);

    private final List<String> allGroups;
    private final MembershipSearchStrategy membershipSearchStrategy;


    protected MembershipsIterableImpl(final DirectoryManager directoryManager,
                                      final SearchStrategyFactory searchStrategyFactory,
                                      final Application application,
                                      final AccessFilterFactory accessFilterFactory) {
        final List<Directory> activeDirectories = getActiveDirectories(application);
        final AccessFilter accessFilter = accessFilterFactory.create(application, true);
        final CanonicalityChecker canonicalityChecker = new OptimizedCanonicalityChecker(directoryManager, activeDirectories);
        this.allGroups = searchStrategyFactory.createGroupSearchStrategy(true, activeDirectories, accessFilter).searchGroups(ALL_GROUPS_QUERY);
        this.membershipSearchStrategy = searchStrategyFactory.createMembershipSearchStrategy(
                application.isMembershipAggregationEnabled(), activeDirectories, canonicalityChecker, accessFilter);
    }

    @Override
    public int groupCount() {
        return allGroups.size();
    }

    @Override
    public Iterator<Membership> iterator() {
        return new AbstractIterator<Membership>() {
            final long start = System.currentTimeMillis();
            final Iterator<List<String>> batchIterator = Iterables.partition(allGroups, GET_MEMBERSHIPS_BATCH_SIZE).iterator();
            Iterator<? extends Membership> current = Collections.emptyIterator();

            @Override
            protected Membership computeNext() {
                while (!current.hasNext() && batchIterator.hasNext()) {
                    current = getMemberships(batchIterator.next());
                }
                if (current.hasNext()) {
                    return current.next();
                }
                logger.debug("Memberships iteration took {}s", (System.currentTimeMillis() - start) / 1000.0);
                return endOfData();
            }
        };
    }

    private Iterator<? extends Membership> getMemberships(final List<String> groupNames) {
        final MembershipQuery<String> userNamesQuery = QueryBuilder.queryFor(String.class, EntityDescriptor.user())
                .childrenOf(EntityDescriptor.group())
                .withNames(groupNames)
                .startingAt(0)
                .returningAtMost(EntityQuery.ALL_RESULTS);
        final MembershipQuery<String> childGroupNames = userNamesQuery.withEntityToReturn(EntityDescriptor.group());

        final ListMultimap<String, String> users = membershipSearchStrategy.searchDirectGroupRelationshipsGroupedByName(userNamesQuery);
        final ListMultimap<String, String> groups = membershipSearchStrategy.searchDirectGroupRelationshipsGroupedByName(childGroupNames);
        return groupNames.stream()
                .map(name -> new ImmutableMembership(name, users.get(name), groups.get(name)))
                .iterator();
    }

    public static ApplicationService.MembershipsIterable runWithClassLoader(final ClassLoader classLoader, final ApplicationService.MembershipsIterable original) {
        final ApplicationService.MembershipsIterable iterable = new ApplicationService.MembershipsIterable() {
            @Override
            public int groupCount() {
                return original.groupCount();
            }

            @Override
            public Iterator<Membership> iterator() {
                return ProxyUtil.runWithContextClassLoader(classLoader, original.iterator());
            }
        };
        return ProxyUtil.runWithContextClassLoader(classLoader, iterable);
    }
}
