package com.atlassian.crowd.manager.application;

import com.atlassian.crowd.embedded.api.Directories;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.impl.IdentifierSet;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.manager.application.canonicality.CanonicalEntityByNameFinder;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.Applications;
import com.atlassian.crowd.model.event.GroupEvent;
import com.atlassian.crowd.model.event.GroupMembershipEvent;
import com.atlassian.crowd.model.event.Operation;
import com.atlassian.crowd.model.event.OperationEvent;
import com.atlassian.crowd.model.event.UserEvent;
import com.atlassian.crowd.model.event.UserMembershipEvent;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Helper class transforming list of {@link OperationEvent} in context of an {@link Application}.
 */
public class EventTransformer {
    private final DirectoryManager directoryManager;
    private final List<Directory> activeDirectories;

    public EventTransformer(final DirectoryManager directoryManager, final Application application) {
        this.directoryManager = directoryManager;
        this.activeDirectories = Applications.getActiveDirectories(application);
    }

    /**
     * Transforms directory events in context of the application. For example:
     * <ul>
     * <li>deleting shadowed user is no-op in context of the application</li>
     * <li>deleting canonical user that is shadowed is an update operation in context of the application</li>
     * </ul>
     */
    public List<OperationEvent> transformEvents(Iterable<OperationEvent> events) throws OperationFailedException {
        final List<OperationEvent> transformed = new ArrayList<>();
        for (final OperationEvent event : events) {
            if (event.getDirectoryId() == null) {
                // events not associated with any directory cannot be masked
                transformed.add(event);
            } else {
                final int eventDirectoryIndex = Iterables.indexOf(activeDirectories,
                        Directories.directoryWithIdPredicate(event.getDirectoryId()));
                if (eventDirectoryIndex == -1) {
                    // event is not in the active directories, do not propagate
                } else if (event instanceof UserEvent) {
                    final UserEvent userEvent = (UserEvent) event;
                    transformed.addAll(processUserEvent(eventDirectoryIndex, userEvent));
                } else if (event instanceof GroupEvent) {
                    final GroupEvent groupEvent = (GroupEvent) event;
                    transformed.addAll(processGroupEvent(eventDirectoryIndex, groupEvent));
                } else if (event instanceof UserMembershipEvent) {
                    final UserMembershipEvent userMembershipEvent = (UserMembershipEvent) event;
                    transformed.addAll(processUserMembershipEvent(userMembershipEvent));
                } else if (event instanceof GroupMembershipEvent) {
                    final GroupMembershipEvent groupMembershipEvent = (GroupMembershipEvent) event;
                    transformed.addAll(processGroupMembershipEvent(groupMembershipEvent));
                } else {
                    throw new IllegalArgumentException("Event type " + event.getClass() + " not supported.");
                }
            }
        }
        return transformed;
    }


    private List<? extends OperationEvent> processUserEvent(int eventDirectoryIndex, UserEvent event) throws OperationFailedException {
        final List<? extends OperationEvent> events;
        final String username = event.getUser().getName();
        final List<Directory> earlierDirectories = activeDirectories.subList(0, eventDirectoryIndex);
        final List<Directory> laterDirectories = activeDirectories.subList(eventDirectoryIndex + 1, activeDirectories.size());
        if (findUser(earlierDirectories, username) != null) { // Masked
            if (event.getOperation() == Operation.DELETED) {
                final Set<String> parentGroupNames = getParentGroupNames(EntityDescriptor.user(), username);
                events = ImmutableList.of(new UserMembershipEvent(Operation.UPDATED, null, username, parentGroupNames));
            } else {
                // Event entity is masked by an entity in earlier directory
                events = ImmutableList.of();
            }
        } else if (event.getOperation() == Operation.CREATED) {
            if (findUser(laterDirectories, username) != null) { // Masking
                events = ImmutableList.of(new UserEvent(Operation.UPDATED,
                        null, // Not used
                        event.getUser(),
                        event.getStoredAttributes(),
                        event.getDeletedAttributes()));
            } else {
                events = ImmutableList.of(event);
            }
        } else if (event.getOperation() == Operation.DELETED) {
            final User laterUser = findUser(laterDirectories, username);
            if (laterUser != null) { // Masking
                final Set<String> parentGroupNames = getParentGroupNames(EntityDescriptor.user(), username);
                final OperationEvent userEvent = new UserEvent(Operation.UPDATED,
                        null, // Not used
                        laterUser,
                        null,
                        null);
                final OperationEvent membershipEvent = new UserMembershipEvent(Operation.UPDATED, null, username, parentGroupNames);
                events = ImmutableList.of(userEvent, membershipEvent);
            } else {
                events = ImmutableList.of(event);
            }
        } else { // Updated
            events = ImmutableList.of(event);
        }
        return events;
    }

    private List<? extends OperationEvent> processGroupEvent(int eventDirectoryIndex, GroupEvent event) throws OperationFailedException {
        final List<? extends OperationEvent> events;
        final String groupName = event.getGroup().getName();
        final List<Directory> earlierDirectories = activeDirectories.subList(0, eventDirectoryIndex);
        if (findGroup(earlierDirectories, groupName) != null) { // Masked
            if (event.getOperation() == Operation.DELETED) {
                final Set<String> parentGroupNames = getParentGroupNames(EntityDescriptor.group(GroupType.GROUP), groupName);
                final Set<String> childGroupNames = getChildGroupNames(groupName);
                events = ImmutableList.of(new GroupMembershipEvent(Operation.UPDATED, null, groupName, parentGroupNames, childGroupNames));
            } else {
                // Event entity is masked by an entity in earlier directory
                events = ImmutableList.of();
            }
        } else if (event.getOperation() == Operation.CREATED) {
            final List<Directory> laterDirectories = activeDirectories.subList(eventDirectoryIndex + 1, activeDirectories.size());
            if (findGroup(laterDirectories, groupName) != null) { // Masking
                events = ImmutableList.of(new GroupEvent(
                        Operation.UPDATED,
                        null, // Not used
                        event.getGroup(),
                        event.getStoredAttributes(),
                        event.getDeletedAttributes()));
            } else {
                events = ImmutableList.of(event);
            }
        } else if (event.getOperation() == Operation.DELETED) {
            final List<Directory> laterDirectories = activeDirectories.subList(eventDirectoryIndex + 1, activeDirectories.size());
            final Group laterGroup = findGroup(laterDirectories, groupName);
            if (laterGroup != null) { // Masking
                final OperationEvent groupEvent = new GroupEvent(
                        Operation.UPDATED,
                        null, // Not used
                        laterGroup,
                        null,
                        null);

                final Set<String> parentGroupNames = getParentGroupNames(EntityDescriptor.group(GroupType.GROUP), groupName);
                final Set<String> childGroupNames = getChildGroupNames(groupName);
                final OperationEvent membershipEvent = new GroupMembershipEvent(Operation.UPDATED, null, groupName, parentGroupNames, childGroupNames);
                events = ImmutableList.of(groupEvent, membershipEvent);
            } else {
                events = ImmutableList.of(event);
            }
        } else { // Updated
            events = ImmutableList.of(event);
        }
        return events;
    }

    private List<OperationEvent> processUserMembershipEvent(UserMembershipEvent event) throws OperationFailedException {
        final OperationEvent applicationEvent;
        if (event.getOperation() == Operation.DELETED) {
            // Remove only if last
            final String username = event.getChildUsername();
            final IdentifierSet disappearedParentGroupNames = IdentifierSet.difference(
                    event.getParentGroupNames(), getParentGroupNames(EntityDescriptor.user(), username));
            applicationEvent = new UserMembershipEvent(Operation.DELETED, event.getDirectoryId(), username, disappearedParentGroupNames);
        } else {
            applicationEvent = event;
        }
        return ImmutableList.of(applicationEvent);
    }

    private List<OperationEvent> processGroupMembershipEvent(GroupMembershipEvent event) throws OperationFailedException {
        final OperationEvent applicationEvent;
        if (event.getOperation() == Operation.DELETED) {
            // Remove only if last
            final String groupName = event.getGroupName();

            final IdentifierSet parentGroupNames = IdentifierSet.difference(
                    event.getParentGroupNames(), getParentGroupNames(EntityDescriptor.group(), groupName));

            final IdentifierSet childGroupNames = IdentifierSet.difference(
                    event.getChildGroupNames(), getChildGroupNames(groupName));

            applicationEvent = new GroupMembershipEvent(Operation.DELETED, event.getDirectoryId(), groupName, parentGroupNames, childGroupNames);
        } else {
            applicationEvent = event;
        }
        return ImmutableList.of(applicationEvent);
    }


    /**
     * Returns parent group names across all directories.
     *
     * @param entityDescriptor  type of the entity
     * @param name              name of the entity
     * @return list of parent group names of the entity
     * @throws OperationFailedException if one of the active directories does not exist
     */
    private Set<String> getParentGroupNames(EntityDescriptor entityDescriptor, String name) throws OperationFailedException {
        final Set<String> parentGroupNames = new HashSet<>();
        for (Directory directory : activeDirectories) {
            try {
                parentGroupNames.addAll(directoryManager.searchDirectGroupRelationships(directory.getId(),
                        QueryBuilder.queryFor(String.class, EntityDescriptor.group(GroupType.GROUP))
                                .parentsOf(entityDescriptor)
                                .withName(name)
                                .returningAtMost(EntityQuery.ALL_RESULTS)));
            } catch (DirectoryNotFoundException e) {
                // Next sync should pick up this change.
                throw new OperationFailedException("Directory has been removed", e);
            }
        }

        return parentGroupNames;
    }

    /**
     * Returns child group names for a group across all directories.
     *
     * @param groupName         name of the group
     * @return list of child group names of the group
     * @throws OperationFailedException if one of the active directories does not exist
     */
    private Set<String> getChildGroupNames(String groupName) throws OperationFailedException {
        final Set<String> childGroupNames = new HashSet<>();
        for (Directory directory : activeDirectories) {
            try {
                childGroupNames.addAll(directoryManager.searchDirectGroupRelationships(directory.getId(),
                        QueryBuilder.queryFor(String.class, EntityDescriptor.group(GroupType.GROUP))
                                .childrenOf(EntityDescriptor.group(GroupType.GROUP))
                                .withName(groupName)
                                .returningAtMost(EntityQuery.ALL_RESULTS)));
            } catch (DirectoryNotFoundException e) {
                // Next sync should pick up this change.
                throw new OperationFailedException("Directory has been removed", e);
            }
        }

        return childGroupNames;
    }

    private User findUser(Iterable<Directory> directories, String username) throws OperationFailedException {
        return new CanonicalEntityByNameFinder(directoryManager, directories)
                .fastFailingFindOptionalUserByName(username)
                .orElse(null);
    }

    private Group findGroup(Iterable<Directory> directories, String groupName) throws OperationFailedException {
        return new CanonicalEntityByNameFinder(directoryManager, directories)
                .fastFailingFindOptionalGroupByName(groupName)
                .orElse(null);
    }
}
