package com.atlassian.crowd.manager.application;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.stream.Collectors;

/**
 * Finds canonical users that have a provided email address.
 * <p>
 * This class is a workaround for KRAK-3542. One cannot simply use {@link ApplicationService#searchUsers} because it
 * can return non-canonical users.
 */
public class CanonicalUsersByEmailFinder {
    private final ApplicationService applicationService;

    public CanonicalUsersByEmailFinder(ApplicationService applicationService) {
        this.applicationService = applicationService;
    }

    @VisibleForTesting
    static EntityQuery<String> candidatesQuery(String email) {
        return QueryBuilder.queryFor(
                String.class, EntityDescriptor.user(),
                Restriction.on(UserTermKeys.EMAIL).exactlyMatching(email),
                0, EntityQuery.ALL_RESULTS);
    }

    @VisibleForTesting
    static EntityQuery<User> usersQuery(List<String> usernames) {
        return QueryBuilder.queryFor(
                User.class, EntityDescriptor.user(),
                Restriction.on(UserTermKeys.USERNAME).exactlyMatchingAny(usernames),
                0, EntityQuery.ALL_RESULTS);
    }

    /**
     * Returns canonical users that have a provided email address.
     *
     * @param application application
     * @param email       email address
     * @return users sharing provided email address, empty list if no users found
     */
    public List<String> findCanonicalUsersByEmail(Application application, String email) {
        final List<String> emailOwners = findEmailOwners(application, email);

        // Due to KRAK-3542, emailOwners list might contain non-canonical users, that's why we need following processing
        return findCanonicalUsersByUsernames(application, emailOwners)
                .stream()
                .filter(user -> IdentifierUtils.equalsInLowerCase(user.getEmailAddress(), email))
                .map(User::getName)
                .collect(Collectors.toList());
    }

    private List<String> findEmailOwners(Application application, String email) {
        return applicationService.searchUsers(application, candidatesQuery(email));
    }

    private List<User> findCanonicalUsersByUsernames(Application application, List<String> usernames) {
        return usernames.isEmpty() ? ImmutableList.of()
                : applicationService.searchUsers(application, usersQuery(usernames));
    }
}
