package com.atlassian.crowd.manager.application;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.atlassian.crowd.cache.UserAuthorisationCache;
import com.atlassian.crowd.event.application.ApplicationUpdatedEvent;
import com.atlassian.crowd.event.directory.DirectoryDeletedEvent;
import com.atlassian.crowd.event.directory.DirectoryUpdatedEvent;
import com.atlassian.crowd.event.migration.XMLRestoreFinishedEvent;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.user.User;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Implementation of ApplicationService which caches the result of ApplicationService methods.
 *
 * @since v2.2
 */
@Transactional
public class CachingApplicationService extends AbstractDelegatingApplicationService {
    private final UserAuthorisationCache userAuthorisationCache;
    private final EventPublisher eventPublisher;
    private static final Logger log = LoggerFactory.getLogger(CachingApplicationService.class);

    public CachingApplicationService(final ApplicationService applicationService, final UserAuthorisationCache userAuthorisationCache, EventPublisher eventPublisher) {
        super(applicationService);
        this.userAuthorisationCache = checkNotNull(userAuthorisationCache);
        this.eventPublisher = eventPublisher;
    }

    @PostConstruct
    public void registerListener() {
        eventPublisher.register(this);
    }

    @PreDestroy
    public void unregisterListener() {
        eventPublisher.unregister(this);
    }

    public boolean isUserAuthorised(final Application application, final String username) {
        try {
            final User user = findUserByName(application, username);
            return isUserAuthorised(application, user);
        } catch (UserNotFoundException e) {
            return false;
        }
    }

    @Override
    public boolean isUserAuthorised(final Application application, final User user) {
        final Boolean allowedToAuthenticate = userAuthorisationCache.isPermitted(user, application.getName());
        if (allowedToAuthenticate != null) {
            return allowedToAuthenticate;
        } else {
            final boolean permitted = getApplicationService().isUserAuthorised(application, user);
            // only cache positive results
            if (permitted) {
                userAuthorisationCache.setPermitted(user, application.getName(), permitted);
            }
            return permitted;
        }
    }

    @EventListener
    public void onApplicationUpdated(ApplicationUpdatedEvent event) {
        clearCache(event);
    }

    @EventListener
    public void onDirectoryUpdated(DirectoryUpdatedEvent event) {
        clearCache(event);
    }

    @EventListener
    public void onDirectoryDeleted(DirectoryDeletedEvent event) {
        clearCache(event);
    }

    @EventListener
    public void onBackupRestored(XMLRestoreFinishedEvent event) {
        clearCache(event);
    }

    private void clearCache(Object event) {
        log.debug("Clearing userAuthorisationCache on {}", event);
        userAuthorisationCache.clear();
    }
}
