package com.atlassian.crowd.manager.application;

import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.embedded.spi.DcLicenseChecker;
import com.atlassian.crowd.event.user.UserAuthenticatedByEmailAddressEvent;
import com.atlassian.crowd.event.user.UserAuthenticationFailedInvalidAuthenticationEvent;
import com.atlassian.crowd.event.user.UserEmailAuthenticationDuplicatedEmailEvent;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.validator.EmailAddressValidator;
import com.atlassian.event.api.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

public class AllowingAuthenticateByEmailApplicationService extends AbstractDelegatingApplicationService {
    private static final Logger logger = LoggerFactory.getLogger(AllowingAuthenticateByEmailApplicationService.class);

    private final EmailAddressValidator emailValidator;
    private final EventPublisher eventPublisher;
    private final CanonicalUsersByEmailFinder canonicalUsersByEmailFinder;
    private final DcLicenseChecker dcLicenseChecker;

    public AllowingAuthenticateByEmailApplicationService(ApplicationService delegate,
                                                         EmailAddressValidator emailValidator,
                                                         EventPublisher eventPublisher,
                                                         CanonicalUsersByEmailFinder canonicalUsersByEmailFinder,
                                                         DcLicenseChecker dcLicenseChecker) {
        super(delegate);
        this.emailValidator = emailValidator;
        this.eventPublisher = eventPublisher;
        this.canonicalUsersByEmailFinder = canonicalUsersByEmailFinder;
        this.dcLicenseChecker = dcLicenseChecker;
    }

    @Override
    public User authenticateUser(Application application, String usernameOrEmail, PasswordCredential passwordCredential)
            throws OperationFailedException, InactiveAccountException, InvalidAuthenticationException,
            ExpiredCredentialException, UserNotFoundException {
        try {
            logger.debug("Trying to authenticate user in application {} by treating {} as username", application.getName(), usernameOrEmail);
            return super.authenticateUser(application, usernameOrEmail, passwordCredential);
        } catch (UserNotFoundException e) {
            if (!application.isAuthenticationViaEmailEnabled() || !dcLicenseChecker.isDcLicense()) {
                throw e;
            }

            logger.debug("User with username {} not found in application {}. Trying to authenticate by treating {} as email",
                    usernameOrEmail, application.getName(), usernameOrEmail);
            return tryToLoginByEmail(application, passwordCredential, usernameOrEmail, e);
        }
    }

    private User tryToLoginByEmail(Application application, PasswordCredential passwordCredential, String email, UserNotFoundException originalException)
            throws InvalidAuthenticationException, OperationFailedException, InactiveAccountException, ExpiredCredentialException,
            UserNotFoundException {
        if (!emailValidator.isValidSyntax(email)) {
            logger.debug("{} is not a valid email. We will not try to authenticate user in app {} by email", email, application.getName());
            throw originalException;
        }

        final List<String> canonicalOwners = canonicalUsersByEmailFinder.findCanonicalUsersByEmail(application, email);

        if (canonicalOwners.isEmpty()) {
            logger.debug("There are no users owning {} email in app {}. Rejecting authentication", email, application.getName());
            throw originalException;
        }

        if (canonicalOwners.size() > 1) {
            logger.debug("There is more than one user in app {} who owns {} email. Rejecting authentication.", application.getName(), email);
            eventPublisher.publish(new UserEmailAuthenticationDuplicatedEmailEvent());
            throw originalException;
        }

        logger.debug("Matched email {} to username {}. Trying to authenticate user", email, canonicalOwners.get(0));

        User authenticatedUser;
        try {
            authenticatedUser = super.authenticateUser(application, canonicalOwners.get(0),
                    passwordCredential);
        } catch (InvalidAuthenticationException e) {
            maybePublishEvent(e);
            throw e;
        }

        eventPublisher.publish(new UserAuthenticatedByEmailAddressEvent());
        return authenticatedUser;
    }

    // Needed to maintain backwards compatibility of ApplicationServiceGeneric event publishing
    private void maybePublishEvent(InvalidAuthenticationException exception) {
        if (exception.getDirectory() != null && exception.getUsername() != null) {
            logger.info("Invalid credentials for user {} in directory {}, aborting", exception.getUsername(),
                    exception.getDirectory().getName());
            eventPublisher.publish(new UserAuthenticationFailedInvalidAuthenticationEvent(this,
                    exception.getDirectory(), exception.getUsername()));
        }
    }
}
