package com.atlassian.crowd.event;

import com.atlassian.crowd.dao.membership.InternalMembershipDao;
import com.atlassian.crowd.dao.tombstone.TombstoneDao;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.embedded.spi.GroupDao;
import com.atlassian.crowd.embedded.spi.UserDao;
import com.atlassian.crowd.event.application.ApplicationDirectoryAddedEvent;
import com.atlassian.crowd.event.application.ApplicationDirectoryOrderUpdatedEvent;
import com.atlassian.crowd.event.application.ApplicationDirectoryRemovedEvent;
import com.atlassian.crowd.event.application.ApplicationUpdatedEvent;
import com.atlassian.crowd.event.directory.DirectoryDeletedEvent;
import com.atlassian.crowd.manager.tombstone.TombstoneManagerImpl;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.Applications;
import com.atlassian.crowd.model.event.AliasEvent;
import com.atlassian.crowd.model.event.GroupEvent;
import com.atlassian.crowd.model.event.GroupMembershipEvent;
import com.atlassian.crowd.model.event.Operation;
import com.atlassian.crowd.model.event.OperationEvent;
import com.atlassian.crowd.model.event.UserEvent;
import com.atlassian.crowd.model.event.UserMembershipEvent;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.ImmutableGroup;
import com.atlassian.crowd.model.membership.InternalMembership;
import com.atlassian.crowd.model.tombstone.AliasTombstone;
import com.atlassian.crowd.model.tombstone.ApplicationUpdatedTombstone;
import com.atlassian.crowd.model.tombstone.EventStreamTombstone;
import com.atlassian.crowd.model.tombstone.GroupMembershipTombstone;
import com.atlassian.crowd.model.tombstone.GroupTombstone;
import com.atlassian.crowd.model.tombstone.UserMembershipTombstone;
import com.atlassian.crowd.model.tombstone.UserTombstone;
import com.atlassian.crowd.model.user.ImmutableUser;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.Combine;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import java.time.Clock;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * A cluster-safe implementation of EventStore that uses the entity creation and update timestamp,
 * and the persisted tombstone information to create an event stream. The produced event streams may
 * be different than the ones produced by @link com.atlassian.crowd.event.EventStoreGeneric,
 * but should lead to creating the same state to the caller replaying the stream (i.e. doing an
 * incremental sync), assuming all operations are idempotent (i.e. adding an entity that's added
 * is not an error, but causes an update; deleting a missing entity is not an error).
 * <p></p>
 * Some caveats:
 * - to alleviate for timestamp skew and operations that are not committed when fetching the events,
 * the implementation produces a sync token that will cause events from a few minutes before the sync to be returned
 * again during the next sync. As long as events are idempotent this shouldn't change the state of the caller.
 * - some events make recreating the event stream impossible, and will throw EventTokenExpiredException as per contract
 * because of the replaying this will make incremental sync impossible for the duration of the backoff
 * - alias events are returned in a simplified form, that's different from EventStoreGeneric - as the only usage
 * is to abort incremental sync when an aliasing event occurs, they're all transformed into alias tombstones, and
 * returned a alias deletions
 * - to limit the size of the list materialized in memory, there's a limit to how many events the implementation will
 * return before throwing EventTokenExpiredException. The implementation MIGHT return larger lists if they are
 * already materialized and there's no point in throwing them away.
 */
public class TimestampBasedEventStore implements EventStore {
    public static final long TIMESTAMP_SLOP_TOLERANCE = TimeUnit.MINUTES.toMillis(5);
    private static final Logger log = LoggerFactory.getLogger(TimestampBasedEventStore.class);

    private final Clock clock;
    private final UserDao userDao;
    private final TombstoneDao tombstoneDao;
    private final GroupDao groupDao;
    private final InternalMembershipDao membershipDao;
    private final int eventCountLimit;

    public TimestampBasedEventStore(UserDao userDao, TombstoneDao tombstoneDao, GroupDao groupDao, InternalMembershipDao membershipDao, int eventCountLimit) {
        this(userDao, groupDao, membershipDao, tombstoneDao, Clock.systemUTC(), eventCountLimit);
    }

    public TimestampBasedEventStore(UserDao userDao, GroupDao groupDao, InternalMembershipDao membershipDao, TombstoneDao tombstoneDao, Clock clock, int eventCountLimit) {
        this.userDao = userDao;
        this.groupDao = groupDao;
        this.membershipDao = membershipDao;
        this.tombstoneDao = tombstoneDao;
        this.clock = clock;
        this.eventCountLimit = eventCountLimit;
        log.debug("Created TimestampBasedEventStore with event limit: {}", eventCountLimit);
    }

    @Override
    public String getCurrentEventToken(List<Long> directoryIds) {
        final long nextCheckTimestamp = clock.millis() - TIMESTAMP_SLOP_TOLERANCE;
        final String marshalledToken = new TimestampBasedEventToken(nextCheckTimestamp, directoryIds).marshall();
        log.debug("Returning event token {}", marshalledToken);
        return marshalledToken;
    }

    @Transactional
    @Override
    public Events getNewEvents(String eventToken, List<Long> directoryIds) throws EventTokenExpiredException {
        log.debug("Got request with token {} for directories: {}", eventToken, directoryIds);
        final TimestampBasedEventToken token = TimestampBasedEventToken.unmarshall(eventToken)
                .orElseThrow(() -> new EventTokenExpiredException("Unrecognized token format"));
        checkIfTokenValid(token, directoryIds);

        long timestamp = token.timestamp;
        final ImmutableList<OperationEvent> events = getDirectoryEvents(directoryIds, timestamp, true);

        return new Events(events, getCurrentEventToken(directoryIds));
    }

    @Transactional
    @Override
    public Events getNewEvents(String eventToken, Application application) throws EventTokenExpiredException {
        log.debug("Got request with token {} for application: {}", eventToken, application);
        final TimestampBasedEventToken token = TimestampBasedEventToken.unmarshall(eventToken)
                .orElseThrow(() -> new EventTokenExpiredException("Unrecognized token format"));
        checkIfTokenValid(token, application);
        final List<Long> directoryIds = Applications.getActiveDirectories(application).stream().map(Directory::getId).collect(Collectors.toList());
        checkIfTokenValid(token, directoryIds);

        long timestamp = token.timestamp;
        final ImmutableList<OperationEvent> events = getDirectoryEvents(directoryIds, timestamp, false);

        return new Events(events, getCurrentEventToken(directoryIds));
    }

    private ImmutableList<OperationEvent> getDirectoryEvents(List<Long> directoryIds, long timestamp, boolean addAliasEvents) throws EventTokenExpiredException {
        final ImmutableList.Builder<OperationEvent> events = ImmutableList.builder();

        // check if any aliases created (or deleted) in the time window, create AliasEvents for those, so that aliasing applications
        // can discard the whole thing and non-aliasing applications can ignore these events
        // these are intentionally not included in the event count limit
        if (addAliasEvents) {
            final List<OperationEvent> syntheticAliasEvents = getAliasEvents(timestamp);
            events.addAll(syntheticAliasEvents);
        }

        int eventCount = 0;
        // get user/groups/memberships added or updated after timestamp
        // get user/groups/memberships removed after timestamp
        for (long directoryId : directoryIds) {
            log.debug("Preparing events for directory {}", directoryIds);

            final EventsHolder userEvents = getUserEvents(timestamp, directoryId);
            checkEventLimit(eventCount += userEvents.size());

            final EventsHolder groupEvents = getGroupEvents(timestamp, directoryId);
            checkEventLimit(eventCount += groupEvents.size());

            final EventsHolder membershipEvents = getMembershipEvents(timestamp, directoryId);
            checkEventLimit(eventCount += membershipEvents.size());

            // we don't guarantee event order, but want to keep ordering so that replaying the sequence will
            // cause the client to be in the same state as the server

            // so removals go before insertions, membership removals are first and membership insertions are last so the users/groups
            // they reference still/already exist
            events
                    .addAll(membershipEvents.removals)
                    .addAll(userEvents.removals)
                    .addAll(groupEvents.removals)
                    .addAll(groupEvents.addsAndUpdates)
                    .addAll(userEvents.addsAndUpdates)
                    .addAll(membershipEvents.addsAndUpdates);
            log.debug("Finished processing directory {}, event count so far: {}", directoryId, eventCount);
        }
        return events.build();
    }

    private void checkIfTokenValid(TimestampBasedEventToken token, Application application) throws EventTokenExpiredException {
        final List<ApplicationUpdatedTombstone> resetTombstones = tombstoneDao.getTombstonesAfter(token.timestamp, application.getId(), ApplicationUpdatedTombstone.class);
        if (!resetTombstones.isEmpty()) {
            log.debug("Found {} application updated tombstones after {}, reporting incremental sync unavailable", resetTombstones.size(), token.timestamp);
            throw new EventTokenExpiredException(String.format("Application configuration has changed"));
        }
        final List<AliasTombstone> aliasTombstones = tombstoneDao.getTombstonesAfter(token.timestamp, application.getId(), AliasTombstone.class);
        if (!aliasTombstones.isEmpty()) {
            log.debug("Found {} alias tombstones after {}, reporting incremental sync unavailable", resetTombstones.size(), token.timestamp);
            throw new EventTokenExpiredException(String.format("Aliasing configuration has changed"));
        }
    }

    private void checkIfTokenValid(TimestampBasedEventToken token, List<Long> directoryIds) throws EventTokenExpiredException {
        if (eventCountLimit <= 0) {
            log.debug("Event count limit is 0, reporting incremental sync unavailable");
            throw new EventTokenExpiredException("Incremental synchronisation is disabled for this server");
        }

        if (!Objects.equals(ImmutableList.copyOf(token.dirIds), ImmutableList.copyOf(directoryIds))) {
            log.debug("Requested events for a different directory set, reporting incremental sync unavailable");
            throw new EventTokenExpiredException("Application configuration has changed");
        }

        if (token.timestamp <= (clock.millis() - TombstoneManagerImpl.TOMBSTONE_LIFETIME.toMillis() / 2)) {
            log.debug("Requested events older than the tombstone expiry threshold, reporting incremental sync unavailable");
            throw new EventTokenExpiredException("The token has expired");
        }

        final List<EventStreamTombstone> resetTombstones = tombstoneDao.getTombstonesAfter(token.timestamp, directoryIds, EventStreamTombstone.class);
        if (!resetTombstones.isEmpty()) {
            log.debug("Found {} event reset tombstones after {}, reporting incremental sync unavailable", resetTombstones.size(), token.timestamp);
            final String firstReason = resetTombstones.get(0).getReason();
            throw new EventTokenExpiredException(String.format("%s is not supported by incremental sync.", firstReason));
        }
    }

    private EventsHolder getUserEvents(long timestamp, long directoryId) throws EventTokenExpiredException {
        log.debug("Getting user events for directory {} since {}", directoryId, timestamp);
        final QueryBuilder.PartialEntityQuery<User> entityQuery = QueryBuilder.queryFor(User.class, EntityDescriptor.user());
        final List<User> removedUsers = tombstoneDao.getTombstonesAfter(timestamp, Collections.singleton(directoryId), UserTombstone.class)
                .stream().map(UserTombstone::toUser).collect(Collectors.toList());

        final EventsHolder userEvents = getEntityStream(directoryId, timestamp, removedUsers, entityQuery, userDao::search);
        log.debug("Got {} user adds/updates and {} user removals", userEvents.addsAndUpdates.size(), userEvents.removals.size());
        return userEvents;
    }

    private EventsHolder getGroupEvents(long timestamp, long directoryId) throws EventTokenExpiredException {
        log.debug("Getting group events for directory {} since {}", directoryId, timestamp);
        final QueryBuilder.PartialEntityQuery<Group> entityQuery = QueryBuilder.queryFor(Group.class, EntityDescriptor.group());
        final List<Group> removedGroups = tombstoneDao.getTombstonesAfter(timestamp, Collections.singleton(directoryId), GroupTombstone.class)
                .stream().map(GroupTombstone::toGroup).collect(Collectors.toList());

        final EventsHolder groupEvents = getEntityStream(directoryId, timestamp, removedGroups, entityQuery, groupDao::search);
        log.debug("Got {} group adds/updates and {} group removals", groupEvents.addsAndUpdates.size(), groupEvents.removals.size());
        return groupEvents;
    }

    private EventsHolder getMembershipEvents(long timestamp, long directoryId) throws EventTokenExpiredException {
        log.debug("Getting membership events for directory {} since {}", directoryId, timestamp);

        final Date cutoffDate = new Date(timestamp);
        final List<UserMembershipTombstone> userMembershipTombstones = tombstoneDao.getTombstonesAfter(timestamp, Collections.singleton(directoryId), UserMembershipTombstone.class);
        checkEventLimit(userMembershipTombstones.size());

        final List<GroupMembershipTombstone> groupMembershipTombstones = tombstoneDao.getTombstonesAfter(timestamp, Collections.singleton(directoryId), GroupMembershipTombstone.class);
        checkEventLimit(groupMembershipTombstones.size() + userMembershipTombstones.size());


        final Set<OperationEvent> addedMemberships = membershipDao.getMembershipsCreatedAfter(directoryId, cutoffDate, eventCountLimit + 1)
                .stream().map(membership -> toEvent(Operation.CREATED, membership)).collect(Collectors.toSet());
        checkEventLimit(groupMembershipTombstones.size() + userMembershipTombstones.size() + addedMemberships.size());

        final Stream<OperationEvent> recreatedMemberships = Stream.concat(
                userMembershipTombstones.stream()
                        .filter(tombstone -> !addedMemberships.contains(new UserMembershipEvent(Operation.CREATED, tombstone.getDirectoryId(), tombstone.getChildName(), tombstone.getParentName())))
                        .filter(tombstone -> membershipDao.isUserDirectMember(directoryId, tombstone.getChildName(), tombstone.getParentName()))
                        .map(membership -> new UserMembershipEvent(Operation.CREATED, membership.getDirectoryId(), membership.getChildName(), membership.getParentName())),
                groupMembershipTombstones.stream()
                        .filter(tombstone -> !addedMemberships.contains(new GroupMembershipEvent(Operation.CREATED, tombstone.getDirectoryId(), tombstone.getChildName(), tombstone.getParentName())))
                        .filter(tombstone -> membershipDao.isGroupDirectMember(directoryId, tombstone.getChildName(), tombstone.getParentName()))
                        .map(membership -> new GroupMembershipEvent(Operation.CREATED, membership.getDirectoryId(), membership.getChildName(), membership.getParentName()))
        );

        final EventsHolder membershipEvents = new EventsHolder(
                Stream.concat(
                        userMembershipTombstones.stream().map(tombstone -> new UserMembershipEvent(Operation.DELETED, directoryId, tombstone.getChildName(), tombstone.getParentName())),
                        groupMembershipTombstones.stream().map(tombstone -> new GroupMembershipEvent(Operation.DELETED, directoryId, tombstone.getChildName(), tombstone.getParentName()))
                ),
                Stream.concat(recreatedMemberships, addedMemberships.stream())
        );
        log.debug("Got {} membership adds/updates and {} membership removals", membershipEvents.addsAndUpdates.size(), membershipEvents.removals.size());
        return membershipEvents;
    }

    private <T extends DirectoryEntity> EventsHolder getEntityStream(long directoryId, long timestamp,
                                                                     List<T> removedEntities,
                                                                     QueryBuilder.PartialEntityQuery<T> entityQuery,
                                                                     BiFunction<Long, EntityQuery<T>, List<T>> searchFunction) throws EventTokenExpiredException {
        checkEventLimit(removedEntities.size());
        final Date cutoffDate = new Date(timestamp);

        final Set<T> created = ImmutableSet.copyOf(searchFunction.apply(directoryId, entityQuery
                .with(Restriction.on(UserTermKeys.CREATED_DATE).greaterThan(cutoffDate))
                .returningAtMost(eventCountLimit + 1)));
        checkEventLimit(removedEntities.size() + created.size());

        // due to timestamp drift these may or may not be deleted, check for the entities we're not sure about
        // to avoid sending a DELETE if the entity does exist in the directory
        final List<SearchRestriction> potentiallyRecreatedEntities = removedEntities.stream()
                .filter(entity -> !created.contains(entity))
                .map(tombstone -> Restriction.on(UserTermKeys.USERNAME).exactlyMatching(tombstone.getName()))
                .collect(Collectors.toList());
        final List<T> recreated = potentiallyRecreatedEntities.isEmpty() ? Collections.emptyList() :
                searchFunction.apply(directoryId, entityQuery
                        .with(Combine.anyOf(potentiallyRecreatedEntities))
                        .returningAtMost(EntityQuery.ALL_RESULTS));

        final Stream<OperationEvent> createdStream = Stream.concat(created.stream(), recreated.stream()).unordered().distinct()
                .map(entity -> toEvent(Operation.CREATED, entity));

        final List<T> updatedEntities = searchFunction.apply(directoryId, entityQuery
                .with(Combine.allOf(Restriction.on(UserTermKeys.UPDATED_DATE).greaterThan(cutoffDate),
                        Combine.anyOf(
                                Restriction.on(UserTermKeys.CREATED_DATE).lessThan(cutoffDate),
                                Restriction.on(UserTermKeys.CREATED_DATE).exactlyMatching(cutoffDate))
                ))
                .returningAtMost(eventCountLimit + 1));
        checkEventLimit(removedEntities.size() + created.size() + updatedEntities.size());

        final Stream<OperationEvent> updatedStream = updatedEntities.stream().map(user -> toEvent(Operation.UPDATED, user));
        final Stream<OperationEvent> removedStream = removedEntities.stream()
                .map(entity -> toEvent(Operation.DELETED, entity));

        return new EventsHolder(removedStream, Stream.concat(createdStream, updatedStream));
    }

    private <T extends DirectoryEntity> OperationEvent toEvent(Operation operation, T entity) {
        if (entity instanceof User) {
            final User user = (User) entity;
            return new UserEvent(operation, user.getDirectoryId(), ImmutableUser.from(user), null, null);
        } else if (entity instanceof Group) {
            final Group group = (Group) entity;
            return new GroupEvent(operation, group.getDirectoryId(), ImmutableGroup.from(group), null, null);
        }

        throw new IllegalArgumentException("Unknown entity type " + entity);
    }

    private OperationEvent toEvent(Operation operation, InternalMembership membership) {
        switch (membership.getMembershipType()) {
            case GROUP_USER:
                return new UserMembershipEvent(operation, membership.getDirectory().getId(), membership.getChildName(), membership.getParentName());
            case GROUP_GROUP:
                return new GroupMembershipEvent(operation, membership.getDirectory().getId(), membership.getChildName(), membership.getParentName());
            default:
                throw new IllegalArgumentException("Unknown membership type " + membership.getMembershipType());
        }
    }

    private List<OperationEvent> getAliasEvents(long timestamp) {
        log.debug("Getting alias events since {}", timestamp);
        final List<AliasTombstone> aliasTombstones = tombstoneDao.getTombstonesAfter(timestamp, Collections.emptySet(), AliasTombstone.class);
        final List<OperationEvent> events = aliasTombstones.stream().map(tomb -> AliasEvent.deleted(tomb.getUsername(), tomb.getApplicationId())).collect(Collectors.toList());
        log.debug("Got {} alias events (tombstones)", events.size());
        return events;
    }

    private void checkEventLimit(int count) throws EventTokenExpiredException {
        if (count > eventCountLimit) {
            log.debug("Failed event limit check ({} of {}), returning failure", count, eventCountLimit);
            throw new EventTokenExpiredException("Too many events since previous incremental sync");
        }
    }

    @Override
    public void storeOperationEvent(OperationEvent event) {
        if (event instanceof AliasEvent) {
            // save all alias events as tombstones - we only use them to invalidate the stream for aliasing applications currently,
            // so no need for distinguishing operations or storing other vales
            final AliasEvent aliasEvent = (AliasEvent) event;
            tombstoneDao.storeAliasTombstone(aliasEvent.getApplicationId(), aliasEvent.getUsername());
        }

        // ignored otherwise
    }

    @Override
    public void handleApplicationEvent(Object event) {
        final Class<?> eventClass = event.getClass();
        if (ApplicationDirectoryRemovedEvent.class.isAssignableFrom(eventClass) ||
                ApplicationDirectoryAddedEvent.class.isAssignableFrom(eventClass) ||
                ApplicationDirectoryOrderUpdatedEvent.class.isAssignableFrom(eventClass) ||
                DirectoryDeletedEvent.class.isAssignableFrom(eventClass)) {
            // ignored as these are handled on a per-sync basis using the directories sequence encoded in the token
            return;
        } else if (event instanceof DirectoryEvent) {
            final Long directoryId = ((DirectoryEvent) event).getDirectoryId();
            log.debug("Storing events tombstone for directory {} because of {}", directoryId, event);
            tombstoneDao.storeEventsTombstoneForDirectory(eventClass.getName(), directoryId);
        } else if (event instanceof ApplicationUpdatedEvent) {
            tombstoneDao.storeEventsTombstoneForApplication(((ApplicationUpdatedEvent)event).getApplicationId());
        } else {
            log.debug("Storing global events tombstone because of {}", event);
            tombstoneDao.storeEventsTombstone(eventClass.getName());
        }

    }

    private static class EventsHolder {
        final Collection<OperationEvent> removals;
        final Collection<OperationEvent> addsAndUpdates;

        public EventsHolder(Stream<OperationEvent> removals, Stream<OperationEvent> addsAndUpdates) {
            this.removals = removals.collect(Collectors.toList());
            this.addsAndUpdates = addsAndUpdates.collect(Collectors.toList());
        }

        public int size() {
            return removals.size() + addsAndUpdates.size();
        }
    }

}