package com.atlassian.crowd.directory.synchronisation.cache;

import com.atlassian.crowd.directory.DirectoryCacheChangeOperations;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.InternalDirectoryGroup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.Date;

import static com.atlassian.crowd.util.EqualityUtil.different;

/**
 * The default GroupActionStrategy, that does not have explicit handling for groups with duplicate names and different
 * external ids
 */
public class DefaultGroupActionStrategy extends AbstractGroupActionStrategy {

    private static final Logger logger = LoggerFactory.getLogger(DefaultGroupActionStrategy.class);

    @Override
    public DirectoryCacheChangeOperations.GroupsToAddUpdateReplace decide(
            @Nullable final InternalDirectoryGroup groupMatchedByName,
            @Nullable final InternalDirectoryGroup groupMatchedByExternalId,
            final Group remoteGroup,
            final Date syncStartDate,
            final long directoryId) throws OperationFailedException {
        if (groupMatchedByName == null) {
            // Group does not exist at this point - we need to create it.
            logger.debug("group [ {} ] not found, adding", remoteGroup.getName());
            return addGroup(remoteGroup);
        }

        if (!remoteGroup.getName().equals(groupMatchedByName.getName())) {
            logger.warn("remote group name [ {} ] casing differs from local group name [ {} ]. Group details will be kept updated, but the group name cannot be updated", remoteGroup.getName(), groupMatchedByName.getName());
        }

        if (wasGroupUpdatedAfterSearchStart(remoteGroup, groupMatchedByName, syncStartDate, directoryId)) {
            return NO_OP;
        }

        if (groupMatchedByName.isLocal()) {
            // Looks like the admin created a Local Group, then a Group was created in the remote LDAP server with same name.
            // We will keep the local group and its members and ignore the remote group.
            logger.info("group [ {} ] in directory [ {} ] matches local group of same name, skipping", remoteGroup.getName(), directoryId);
            return NO_OP;
        }
        // The group already exists in the Local Directory cache, we may need to update it.
        // First we need to compare the GroupTypes of the two groups as this will require special handling.
        if (remoteGroup.getType() == GroupType.LEGACY_ROLE && groupMatchedByName.getType() == GroupType.GROUP) {
            // Ignore the incoming Role because we have a Group in the cache. Group has precedence.
            logger.debug("role [ {} ] in directory [ {} ] matches local group of same name, skipping", remoteGroup.getName(), directoryId);
            return NO_OP;
        }
        if (remoteGroup.getType() == GroupType.GROUP && groupMatchedByName.getType() == GroupType.LEGACY_ROLE) {
            // Let the incoming Group override the Role that is currently in the cache
            logger.debug("role [ {} ] in directory [ {} ] matches legacy role of same name, replacing", groupMatchedByName.getName(), directoryId);
            // We can't just do an update of GroupType - we must delete the Role and insert a new Group
            return replaceGroup(groupMatchedByName, remoteGroup);
        }
        // GroupTypes are the same - check if any other values need updating (ie description)
        if (hasChanged(remoteGroup, groupMatchedByName)) {
            return updateGroup(groupMatchedByName, remoteGroup);
        }

        // Group has not changed
        logger.trace("group [ {} ] unmodified, skipping", remoteGroup.getName());
        return NO_OP;
    }

    private static boolean hasChanged(final Group remoteGroup, final Group internalGroup) {
        return different(remoteGroup.getDescription(), internalGroup.getDescription())
                || different(remoteGroup.getExternalId(), internalGroup.getExternalId());
    }
}
